<?php
/**
 * Image Optimizer Class
 *
 * @package EnBombas\Optimizer
 */

namespace EnBombas\Optimizer;

use EnBombas\Compression\Compressor;
use EnBombas\Backup\BackupManager;
use EnBombas\Core\Plugin;

/**
 * Class ImageOptimizer
 *
 * Main image optimization orchestrator.
 */
class ImageOptimizer {
    /**
     * Compressor instance
     */
    private Compressor $compressor;

    /**
     * Backup manager instance
     */
    private BackupManager $backup_manager;

    /**
     * Constructor
     */
    public function __construct(BackupManager $backup_manager) {
        $this->compressor = new Compressor();
        $this->backup_manager = $backup_manager;
    }

    /**
     * Optimize an attachment
     *
     * @param int $attachment_id The attachment ID
     * @return array Result with success status and data
     */
    public function optimize(int $attachment_id): array {
        $plugin = Plugin::get_instance();
        $settings = $plugin->get_settings();

        // Check if already optimized
        if ($this->is_optimized($attachment_id)) {
            return [
                'success' => false,
                'message' => __('Image is already optimized.', 'en-bombas'),
                'already_optimized' => true,
            ];
        }

        // Get attachment file path
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return [
                'success' => false,
                'message' => __('Attachment file not found.', 'en-bombas'),
            ];
        }

        // Check if it's an image
        if (!wp_attachment_is_image($attachment_id)) {
            // Check for PDF
            $mime = get_post_mime_type($attachment_id);
            if ($mime !== 'application/pdf') {
                return [
                    'success' => false,
                    'message' => __('Not an optimizable file type.', 'en-bombas'),
                ];
            }
        }

        // Create backup if enabled
        if ($settings['backup_enabled']) {
            $backup_result = $this->backup_manager->backup($attachment_id);
            if (!$backup_result['success']) {
                return [
                    'success' => false,
                    'message' => __('Failed to create backup.', 'en-bombas'),
                ];
            }
        }

        // Configure compressor
        $this->compressor
            ->set_level($settings['compression_level'])
            ->set_remove_exif($settings['remove_exif']);

        // Resize if needed
        if ($settings['resize_larger']) {
            $this->resize_if_needed($file_path, $settings['max_width'], $settings['max_height']);
        }

        // Compress main image
        $result = $this->compressor->compress($file_path);

        if (!$result['success']) {
            // Restore backup on failure
            if ($settings['backup_enabled']) {
                $this->backup_manager->restore($attachment_id);
            }
            return $result;
        }

        $total_saved = $result['saved_bytes'];

        // Optimize thumbnails if enabled
        if ($settings['optimize_thumbnails']) {
            $thumbnails_result = $this->optimize_thumbnails($attachment_id);
            $total_saved += $thumbnails_result['saved_bytes'];
        }

        // Update attachment metadata
        $this->save_optimization_data($attachment_id, [
            'original_size' => $result['original_size'],
            'optimized_size' => $result['new_size'],
            'saved_bytes' => $total_saved,
            'saved_percentage' => $result['saved_percentage'],
            'compression_level' => $result['compression_level'],
            'optimized_at' => current_time('mysql'),
        ]);

        // Update global stats
        $this->update_global_stats($total_saved);

        return [
            'success' => true,
            'message' => sprintf(
                __('Image optimized! Saved %s (%s%%)', 'en-bombas'),
                size_format($total_saved),
                $result['saved_percentage']
            ),
            'saved_bytes' => $total_saved,
            'saved_percentage' => $result['saved_percentage'],
            'compression_level' => $result['compression_level'],
        ];
    }

    /**
     * Optimize all thumbnails for an attachment
     */
    private function optimize_thumbnails(int $attachment_id): array {
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!$metadata || empty($metadata['sizes'])) {
            return ['success' => true, 'saved_bytes' => 0];
        }

        $upload_dir = wp_upload_dir();
        $base_dir = dirname(get_attached_file($attachment_id));
        $total_saved = 0;

        foreach ($metadata['sizes'] as $size => $data) {
            $thumb_path = $base_dir . '/' . $data['file'];

            if (!file_exists($thumb_path)) {
                continue;
            }

            $result = $this->compressor->compress($thumb_path);

            if ($result['success']) {
                $total_saved += $result['saved_bytes'];
            }
        }

        return [
            'success' => true,
            'saved_bytes' => $total_saved,
        ];
    }

    /**
     * Resize image if it exceeds maximum dimensions
     */
    private function resize_if_needed(string $file_path, int $max_width, int $max_height): bool {
        $image_data = @getimagesize($file_path);
        if (!$image_data) {
            return false;
        }

        $width = $image_data[0];
        $height = $image_data[1];

        // Check if resize is needed
        if ($width <= $max_width && $height <= $max_height) {
            return false;
        }

        // Calculate new dimensions maintaining aspect ratio
        $ratio = min($max_width / $width, $max_height / $height);
        $new_width = (int) round($width * $ratio);
        $new_height = (int) round($height * $ratio);

        // Use WordPress image editor
        $editor = wp_get_image_editor($file_path);
        if (is_wp_error($editor)) {
            return false;
        }

        $editor->resize($new_width, $new_height, false);
        $result = $editor->save($file_path);

        return !is_wp_error($result);
    }

    /**
     * Check if attachment is already optimized
     */
    public function is_optimized(int $attachment_id): bool {
        return (bool) get_post_meta($attachment_id, '_enbombas_optimized', true);
    }

    /**
     * Get optimization data for an attachment
     */
    public function get_optimization_data(int $attachment_id): ?array {
        $data = get_post_meta($attachment_id, '_enbombas_data', true);
        return is_array($data) ? $data : null;
    }

    /**
     * Save optimization data for an attachment
     */
    private function save_optimization_data(int $attachment_id, array $data): void {
        update_post_meta($attachment_id, '_enbombas_optimized', '1');
        update_post_meta($attachment_id, '_enbombas_data', $data);
        update_post_meta($attachment_id, '_enbombas_saved_bytes', $data['saved_bytes']);
    }

    /**
     * Clear optimization data for an attachment
     */
    public function clear_optimization_data(int $attachment_id): void {
        delete_post_meta($attachment_id, '_enbombas_optimized');
        delete_post_meta($attachment_id, '_enbombas_data');
        delete_post_meta($attachment_id, '_enbombas_saved_bytes');
        delete_post_meta($attachment_id, '_enbombas_webp');
        delete_post_meta($attachment_id, '_enbombas_avif');
    }

    /**
     * Update global statistics
     */
    private function update_global_stats(int $saved_bytes): void {
        $plugin = Plugin::get_instance();
        $current_total = $plugin->get_setting('total_saved_bytes', 0);
        $current_count = $plugin->get_setting('total_optimized', 0);

        $plugin->update_setting('total_saved_bytes', $current_total + $saved_bytes);
        $plugin->update_setting('total_optimized', $current_count + 1);
    }

    /**
     * Get compressor instance
     */
    public function get_compressor(): Compressor {
        return $this->compressor;
    }

    /**
     * Optimize a file from custom folder (non-attachment)
     */
    public function optimize_file(string $file_path): array {
        if (!file_exists($file_path)) {
            return [
                'success' => false,
                'message' => __('File not found.', 'en-bombas'),
            ];
        }

        $plugin = Plugin::get_instance();
        $settings = $plugin->get_settings();

        // Configure compressor
        $this->compressor
            ->set_level($settings['compression_level'])
            ->set_remove_exif($settings['remove_exif']);

        // Compress the file
        return $this->compressor->compress($file_path);
    }

    /**
     * Get list of unoptimized attachments
     */
    public function get_unoptimized_attachments(int $limit = -1): array {
        global $wpdb;

        $query = "
            SELECT p.ID 
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_enbombas_optimized'
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE 'image/%'
            AND (pm.meta_value IS NULL OR pm.meta_value != '1')
        ";

        if ($limit > 0) {
            $query .= $wpdb->prepare(" LIMIT %d", $limit);
        }

        return $wpdb->get_col($query);
    }

    /**
     * Get count of unoptimized attachments
     */
    public function get_unoptimized_count(): int {
        global $wpdb;

        return (int) $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_enbombas_optimized'
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE 'image/%'
            AND (pm.meta_value IS NULL OR pm.meta_value != '1')
        ");
    }
}



