<?php
/**
 * Preload & Prefetch
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class Preload
 *
 * Handles resource preloading, prefetching, and preconnecting.
 */
class Preload {
    /**
     * Nitro core instance
     */
    private NitroCore $nitro;

    /**
     * Constructor
     */
    public function __construct(NitroCore $nitro) {
        $this->nitro = $nitro;
    }

    /**
     * Add preload/prefetch hints to HTML
     */
    public function add_hints(string $html): string {
        $hints = [];

        // Preload CSS
        $preload_css = $this->nitro->get_setting('preload_css', []);
        foreach ($preload_css as $url) {
            if (empty($url)) continue;
            $hints[] = sprintf(
                '<link rel="preload" href="%s" as="style">',
                esc_url($url)
            );
        }

        // Preload fonts
        $preload_fonts = $this->nitro->get_setting('preload_fonts', []);
        foreach ($preload_fonts as $url) {
            if (empty($url)) continue;
            
            // Determine font type from extension
            $type = $this->get_font_type($url);
            
            $hints[] = sprintf(
                '<link rel="preload" href="%s" as="font" type="%s" crossorigin>',
                esc_url($url),
                $type
            );
        }

        // Preconnect domains
        $preconnect = $this->nitro->get_setting('preconnect_domains', []);
        foreach ($preconnect as $domain) {
            if (empty($domain)) continue;
            
            // Ensure it's a valid domain
            $domain = $this->normalize_domain($domain);
            if (!$domain) continue;

            $hints[] = sprintf('<link rel="preconnect" href="%s">', esc_url($domain));
            $hints[] = sprintf('<link rel="preconnect" href="%s" crossorigin>', esc_url($domain));
        }

        // DNS prefetch
        $dns_prefetch = $this->nitro->get_setting('dns_prefetch', []);
        foreach ($dns_prefetch as $domain) {
            if (empty($domain)) continue;
            
            $domain = $this->normalize_domain($domain);
            if (!$domain) continue;

            $hints[] = sprintf('<link rel="dns-prefetch" href="%s">', esc_url($domain));
        }

        // Add common preconnects automatically
        $auto_preconnect = $this->detect_third_party_domains($html);
        foreach ($auto_preconnect as $domain) {
            // Skip if already added
            $domain_pattern = preg_quote($domain, '/');
            $already_added = false;
            foreach ($hints as $hint) {
                if (preg_match('/' . $domain_pattern . '/', $hint)) {
                    $already_added = true;
                    break;
                }
            }
            if (!$already_added) {
                $hints[] = sprintf('<link rel="preconnect" href="%s" crossorigin>', esc_url($domain));
            }
        }

        if (empty($hints)) {
            return $html;
        }

        // Inject hints after <head>
        $hints_html = "\n<!-- En Bombas Nitro Preload Hints -->\n" . implode("\n", $hints) . "\n";

        if (preg_match('/<head[^>]*>/i', $html, $match)) {
            $html = str_replace($match[0], $match[0] . $hints_html, $html);
        }

        return $html;
    }

    /**
     * Get font MIME type from URL
     */
    private function get_font_type(string $url): string {
        $extension = strtolower(pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION));

        $types = [
            'woff2' => 'font/woff2',
            'woff' => 'font/woff',
            'ttf' => 'font/ttf',
            'otf' => 'font/otf',
            'eot' => 'application/vnd.ms-fontobject',
        ];

        return $types[$extension] ?? 'font/woff2';
    }

    /**
     * Normalize domain URL
     */
    private function normalize_domain(string $domain): ?string {
        $domain = trim($domain);

        // Add https if no protocol
        if (strpos($domain, '//') === false) {
            $domain = 'https://' . $domain;
        }

        // Parse and validate
        $parsed = parse_url($domain);
        if (!isset($parsed['host'])) {
            return null;
        }

        $scheme = $parsed['scheme'] ?? 'https';
        return $scheme . '://' . $parsed['host'];
    }

    /**
     * Detect third-party domains in HTML for auto-preconnect
     */
    private function detect_third_party_domains(string $html): array {
        $domains = [];
        $site_host = parse_url(home_url(), PHP_URL_HOST);

        // Common third-party patterns to preconnect
        $patterns = [
            'fonts.googleapis.com',
            'fonts.gstatic.com',
            'www.google-analytics.com',
            'www.googletagmanager.com',
            'connect.facebook.net',
            'platform.twitter.com',
            'cdn.jsdelivr.net',
            'cdnjs.cloudflare.com',
            'unpkg.com',
            'www.youtube.com',
            'i.ytimg.com',
        ];

        foreach ($patterns as $pattern) {
            if (strpos($html, $pattern) !== false) {
                $domains[] = 'https://' . $pattern;
            }
        }

        // Also extract domains from external resources
        preg_match_all('/(?:src|href)=["\']https?:\/\/([^"\'\/]+)/i', $html, $matches);
        if (!empty($matches[1])) {
            foreach ($matches[1] as $host) {
                // Skip same-site domains
                if ($host === $site_host || strpos($host, $site_host) !== false) {
                    continue;
                }
                
                $domain = 'https://' . $host;
                if (!in_array($domain, $domains)) {
                    $domains[] = $domain;
                }
            }
        }

        // Limit to most important domains
        return array_slice(array_unique($domains), 0, 10);
    }

    /**
     * Register preload hints via wp_head (alternative method)
     */
    public function register_wp_head_hints(): void {
        // Preload CSS
        $preload_css = $this->nitro->get_setting('preload_css', []);
        foreach ($preload_css as $url) {
            if (empty($url)) continue;
            echo sprintf(
                '<link rel="preload" href="%s" as="style">' . "\n",
                esc_url($url)
            );
        }

        // Preload fonts
        $preload_fonts = $this->nitro->get_setting('preload_fonts', []);
        foreach ($preload_fonts as $url) {
            if (empty($url)) continue;
            $type = $this->get_font_type($url);
            echo sprintf(
                '<link rel="preload" href="%s" as="font" type="%s" crossorigin>' . "\n",
                esc_url($url),
                $type
            );
        }

        // Preconnect
        $preconnect = $this->nitro->get_setting('preconnect_domains', []);
        foreach ($preconnect as $domain) {
            if (empty($domain)) continue;
            $domain = $this->normalize_domain($domain);
            if (!$domain) continue;
            echo sprintf('<link rel="preconnect" href="%s">' . "\n", esc_url($domain));
            echo sprintf('<link rel="preconnect" href="%s" crossorigin>' . "\n", esc_url($domain));
        }

        // DNS Prefetch
        $dns_prefetch = $this->nitro->get_setting('dns_prefetch', []);
        foreach ($dns_prefetch as $domain) {
            if (empty($domain)) continue;
            $domain = $this->normalize_domain($domain);
            if (!$domain) continue;
            echo sprintf('<link rel="dns-prefetch" href="%s">' . "\n", esc_url($domain));
        }
    }

    /**
     * Get preload suggestions based on current page
     */
    public function get_suggestions(): array {
        $suggestions = [
            'fonts' => [],
            'css' => [],
            'domains' => [],
        ];

        // Suggest Google Fonts if detected
        if (class_exists('WP_Theme_JSON_Resolver')) {
            $theme_json = \WP_Theme_JSON_Resolver::get_theme_data()->get_settings();
            if (isset($theme_json['typography']['fontFamilies'])) {
                $suggestions['domains'][] = 'fonts.googleapis.com';
                $suggestions['domains'][] = 'fonts.gstatic.com';
            }
        }

        // Suggest common CDNs
        $common_domains = [
            'fonts.googleapis.com' => __('Google Fonts', 'en-bombas'),
            'fonts.gstatic.com' => __('Google Fonts (archivos)', 'en-bombas'),
            'cdnjs.cloudflare.com' => __('Cloudflare CDN', 'en-bombas'),
            'cdn.jsdelivr.net' => __('jsDelivr CDN', 'en-bombas'),
        ];

        $suggestions['common_domains'] = $common_domains;

        return $suggestions;
    }
}



