<?php
/**
 * Nitro Core - Main Performance Module
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class NitroCore
 *
 * Main controller for the Nitro performance module.
 */
class NitroCore {
    /**
     * Singleton instance
     */
    private static ?NitroCore $instance = null;

    /**
     * Nitro settings
     */
    private array $settings = [];

    /**
     * Module instances
     */
    private ?CSSOptimizer $css_optimizer = null;
    private ?JSOptimizer $js_optimizer = null;
    private ?CacheSystem $cache_system = null;
    private ?LazyLoad $lazyload = null;
    private ?Preload $preload = null;
    private ?Database $database = null;
    private ?Heartbeat $heartbeat = null;

    /**
     * Private constructor
     */
    private function __construct() {
        $this->load_settings();
    }

    /**
     * Get singleton instance
     */
    public static function get_instance(): NitroCore {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Load settings
     */
    private function load_settings(): void {
        $defaults = self::get_default_settings();
        $saved = get_option('enbombas_nitro_settings', []);
        $this->settings = wp_parse_args($saved, $defaults);
    }

    /**
     * Get default settings
     */
    public static function get_default_settings(): array {
        return [
            // CSS Optimization
            'css_minify' => false,
            'css_combine' => false,
            'css_async' => false,
            'css_async_method' => 'preload', // preload, media_print
            'css_remove_unused' => false,
            'critical_css' => '',
            'css_exclude' => [],

            // JS Optimization
            'js_minify' => false,
            'js_combine' => false,
            'js_defer' => false,
            'js_delay' => false,
            'js_delay_timeout' => 5000,
            'js_exclude' => [],

            // Cache
            'cache_enabled' => false,
            'cache_mobile' => false,
            'cache_logged_in' => false,
            'cache_exclude_urls' => [],
            'cache_exclude_cookies' => [],
            'cache_lifespan' => 604800, // 7 days

            // LazyLoad
            'lazyload_images' => false,
            'lazyload_iframes' => false,
            'lazyload_youtube' => false,
            'lazyload_threshold' => 300,
            'lazyload_exclude' => [],

            // Preload
            'preload_css' => [],
            'preload_fonts' => [],
            'preconnect_domains' => [],
            'dns_prefetch' => [],

            // Database
            'db_clean_revisions' => false,
            'db_clean_drafts' => false,
            'db_clean_trash' => false,
            'db_clean_transients' => false,
            'db_optimize_tables' => false,
            'db_auto_clean' => false,
            'db_auto_clean_frequency' => 'weekly',

            // Heartbeat
            'heartbeat_behavior' => 'default', // default, reduce, disable
            'heartbeat_frequency' => 60,
            'heartbeat_backend_only' => false,
        ];
    }

    /**
     * Get setting
     */
    public function get_setting(string $key, mixed $default = null): mixed {
        return $this->settings[$key] ?? $default;
    }

    /**
     * Get all settings
     */
    public function get_settings(): array {
        return $this->settings;
    }

    /**
     * Update settings
     */
    public function update_settings(array $settings): void {
        $this->settings = wp_parse_args($settings, $this->settings);
        update_option('enbombas_nitro_settings', $this->settings);
    }

    /**
     * Initialize the module
     */
    public function init(): void {
        // Initialize sub-modules
        $this->css_optimizer = new CSSOptimizer($this);
        $this->js_optimizer = new JSOptimizer($this);
        $this->cache_system = new CacheSystem($this);
        $this->lazyload = new LazyLoad($this);
        $this->preload = new Preload($this);
        $this->database = new Database($this);
        $this->heartbeat = new Heartbeat($this);

        // Register hooks
        $this->register_hooks();
    }

    /**
     * Register hooks
     */
    private function register_hooks(): void {
        // AJAX handlers
        add_action('wp_ajax_enbombas_nitro_save', [$this, 'ajax_save_settings']);
        add_action('wp_ajax_enbombas_nitro_clear_cache', [$this, 'ajax_clear_cache']);
        add_action('wp_ajax_enbombas_nitro_optimize_db', [$this, 'ajax_optimize_database']);
        add_action('wp_ajax_enbombas_nitro_get_stats', [$this, 'ajax_get_stats']);

        // Cache invalidation hooks
        add_action('save_post', [$this->cache_system, 'clear_post_cache'], 10, 1);
        add_action('comment_post', [$this->cache_system, 'clear_post_cache'], 10, 1);
        add_action('upgrader_process_complete', [$this->cache_system, 'clear_all_cache']);
        add_action('switch_theme', [$this->cache_system, 'clear_all_cache']);

        // Frontend optimizations (only if not admin)
        if (!is_admin()) {
            // Start output buffering early
            add_action('template_redirect', [$this, 'start_optimization'], 1);
        }

        // Enqueue frontend scripts
        add_action('wp_enqueue_scripts', [$this, 'enqueue_frontend_scripts'], 1);

        // Cron for auto database cleanup
        if ($this->get_setting('db_auto_clean')) {
            add_action('enbombas_nitro_db_cleanup', [$this->database, 'run_cleanup']);
            if (!wp_next_scheduled('enbombas_nitro_db_cleanup')) {
                $frequency = $this->get_setting('db_auto_clean_frequency', 'weekly');
                wp_schedule_event(time(), $frequency, 'enbombas_nitro_db_cleanup');
            }
        }
    }

    /**
     * Start output buffering for optimization
     */
    public function start_optimization(): void {
        // Skip for certain conditions
        if (is_admin() || is_feed() || is_preview() || wp_doing_ajax()) {
            return;
        }

        // Skip for logged-in users if setting is disabled
        if (is_user_logged_in() && !$this->get_setting('cache_logged_in')) {
            // Still apply optimizations but don't cache
            ob_start([$this, 'optimize_output']);
            return;
        }

        // Try to serve cached version first
        if ($this->get_setting('cache_enabled')) {
            $cached = $this->cache_system->serve_cache();
            if ($cached) {
                exit;
            }
        }

        // Start buffering for optimization
        ob_start([$this, 'optimize_output']);
    }

    /**
     * Optimize HTML output
     */
    public function optimize_output(string $html): string {
        if (empty($html)) {
            return $html;
        }

        // Skip non-HTML content
        if (strpos($html, '<html') === false && strpos($html, '<!DOCTYPE') === false) {
            return $html;
        }

        // Apply CSS optimizations
        if ($this->get_setting('css_minify') || $this->get_setting('css_combine') || $this->get_setting('css_async')) {
            $html = $this->css_optimizer->optimize($html);
        }

        // Apply JS optimizations
        if ($this->get_setting('js_minify') || $this->get_setting('js_combine') || $this->get_setting('js_defer') || $this->get_setting('js_delay')) {
            $html = $this->js_optimizer->optimize($html);
        }

        // Apply LazyLoad
        if ($this->get_setting('lazyload_images') || $this->get_setting('lazyload_iframes')) {
            $html = $this->lazyload->optimize($html);
        }

        // Add preload/prefetch tags
        $html = $this->preload->add_hints($html);

        // Add critical CSS
        if (!empty($this->get_setting('critical_css'))) {
            $html = $this->css_optimizer->inject_critical_css($html);
        }

        // Cache the result
        if ($this->get_setting('cache_enabled') && !is_user_logged_in()) {
            $this->cache_system->save_cache($html);
        }

        return $html;
    }

    /**
     * Enqueue frontend scripts
     */
    public function enqueue_frontend_scripts(): void {
        if (is_admin()) {
            return;
        }

        $needs_script = $this->get_setting('lazyload_images') || 
                        $this->get_setting('lazyload_iframes') || 
                        $this->get_setting('js_delay');

        if ($needs_script) {
            wp_enqueue_script(
                'enbombas-nitro-frontend',
                ENBOMBAS_PLUGIN_URL . 'assets/js/nitro-frontend.js',
                [],
                ENBOMBAS_VERSION,
                false // Load in header for lazyload
            );

            wp_localize_script('enbombas-nitro-frontend', 'enbombasNitro', [
                'lazyload' => [
                    'threshold' => $this->get_setting('lazyload_threshold', 300),
                    'images' => $this->get_setting('lazyload_images'),
                    'iframes' => $this->get_setting('lazyload_iframes'),
                    'youtube' => $this->get_setting('lazyload_youtube'),
                ],
                'delayJS' => [
                    'enabled' => $this->get_setting('js_delay'),
                    'timeout' => $this->get_setting('js_delay_timeout', 5000),
                ],
            ]);
        }
    }

    /**
     * AJAX: Save settings
     */
    public function ajax_save_settings(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permiso denegado.', 'en-bombas')]);
        }

        $settings = [];

        // CSS Settings
        $settings['css_minify'] = isset($_POST['css_minify']);
        $settings['css_combine'] = isset($_POST['css_combine']);
        $settings['css_async'] = isset($_POST['css_async']);
        $settings['css_async_method'] = sanitize_text_field($_POST['css_async_method'] ?? 'preload');
        $settings['css_remove_unused'] = isset($_POST['css_remove_unused']);
        $settings['critical_css'] = wp_unslash($_POST['critical_css'] ?? '');
        $settings['css_exclude'] = $this->sanitize_textarea_array($_POST['css_exclude'] ?? '');

        // JS Settings
        $settings['js_minify'] = isset($_POST['js_minify']);
        $settings['js_combine'] = isset($_POST['js_combine']);
        $settings['js_defer'] = isset($_POST['js_defer']);
        $settings['js_delay'] = isset($_POST['js_delay']);
        $settings['js_delay_timeout'] = absint($_POST['js_delay_timeout'] ?? 5000);
        $settings['js_exclude'] = $this->sanitize_textarea_array($_POST['js_exclude'] ?? '');

        // Cache Settings
        $settings['cache_enabled'] = isset($_POST['cache_enabled']);
        $settings['cache_mobile'] = isset($_POST['cache_mobile']);
        $settings['cache_logged_in'] = isset($_POST['cache_logged_in']);
        $settings['cache_exclude_urls'] = $this->sanitize_textarea_array($_POST['cache_exclude_urls'] ?? '');
        $settings['cache_lifespan'] = absint($_POST['cache_lifespan'] ?? 604800);

        // LazyLoad Settings
        $settings['lazyload_images'] = isset($_POST['lazyload_images']);
        $settings['lazyload_iframes'] = isset($_POST['lazyload_iframes']);
        $settings['lazyload_youtube'] = isset($_POST['lazyload_youtube']);
        $settings['lazyload_threshold'] = absint($_POST['lazyload_threshold'] ?? 300);
        $settings['lazyload_exclude'] = $this->sanitize_textarea_array($_POST['lazyload_exclude'] ?? '');

        // Preload Settings
        $settings['preload_css'] = $this->sanitize_textarea_array($_POST['preload_css'] ?? '');
        $settings['preload_fonts'] = $this->sanitize_textarea_array($_POST['preload_fonts'] ?? '');
        $settings['preconnect_domains'] = $this->sanitize_textarea_array($_POST['preconnect_domains'] ?? '');
        $settings['dns_prefetch'] = $this->sanitize_textarea_array($_POST['dns_prefetch'] ?? '');

        // Database Settings
        $settings['db_clean_revisions'] = isset($_POST['db_clean_revisions']);
        $settings['db_clean_drafts'] = isset($_POST['db_clean_drafts']);
        $settings['db_clean_trash'] = isset($_POST['db_clean_trash']);
        $settings['db_clean_transients'] = isset($_POST['db_clean_transients']);
        $settings['db_optimize_tables'] = isset($_POST['db_optimize_tables']);
        $settings['db_auto_clean'] = isset($_POST['db_auto_clean']);
        $settings['db_auto_clean_frequency'] = sanitize_text_field($_POST['db_auto_clean_frequency'] ?? 'weekly');

        // Heartbeat Settings
        $settings['heartbeat_behavior'] = sanitize_text_field($_POST['heartbeat_behavior'] ?? 'default');
        $settings['heartbeat_frequency'] = absint($_POST['heartbeat_frequency'] ?? 60);
        $settings['heartbeat_backend_only'] = isset($_POST['heartbeat_backend_only']);

        $this->update_settings($settings);

        // Clear cache when settings change
        $this->cache_system->clear_all_cache();

        wp_send_json_success(['message' => __('¡Configuración guardada, parce! El caché ha sido limpiado.', 'en-bombas')]);
    }

    /**
     * AJAX: Clear cache
     */
    public function ajax_clear_cache(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permiso denegado.', 'en-bombas')]);
        }

        $result = $this->cache_system->clear_all_cache();

        if ($result['success']) {
            wp_send_json_success([
                'message' => sprintf(
                    __('¡Listo, parce! Se limpiaron %d archivos de caché.', 'en-bombas'),
                    $result['files_deleted']
                )
            ]);
        } else {
            wp_send_json_error(['message' => $result['message']]);
        }
    }

    /**
     * AJAX: Optimize database
     */
    public function ajax_optimize_database(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permiso denegado.', 'en-bombas')]);
        }

        $result = $this->database->run_cleanup();

        wp_send_json_success([
            'message' => __('¡Base de datos optimizada, vea pues!', 'en-bombas'),
            'stats' => $result
        ]);
    }

    /**
     * AJAX: Get stats
     */
    public function ajax_get_stats(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permiso denegado.', 'en-bombas')]);
        }

        $stats = [
            'cache' => $this->cache_system->get_stats(),
            'database' => $this->database->get_stats(),
        ];

        wp_send_json_success($stats);
    }

    /**
     * Sanitize textarea to array
     */
    private function sanitize_textarea_array(string $text): array {
        if (empty($text)) {
            return [];
        }
        $lines = explode("\n", $text);
        $lines = array_map('trim', $lines);
        $lines = array_map('sanitize_text_field', $lines);
        return array_filter($lines);
    }

    /**
     * Get CSS optimizer
     */
    public function get_css_optimizer(): CSSOptimizer {
        return $this->css_optimizer;
    }

    /**
     * Get JS optimizer
     */
    public function get_js_optimizer(): JSOptimizer {
        return $this->js_optimizer;
    }

    /**
     * Get cache system
     */
    public function get_cache_system(): CacheSystem {
        return $this->cache_system;
    }

    /**
     * Get database optimizer
     */
    public function get_database(): Database {
        return $this->database;
    }
}



