<?php
/**
 * LazyLoad
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class LazyLoad
 *
 * Handles lazy loading of images, iframes, and YouTube embeds.
 */
class LazyLoad {
    /**
     * Nitro core instance
     */
    private NitroCore $nitro;

    /**
     * Placeholder image (1x1 transparent GIF)
     */
    private const PLACEHOLDER = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';

    /**
     * Constructor
     */
    public function __construct(NitroCore $nitro) {
        $this->nitro = $nitro;
    }

    /**
     * Optimize HTML for lazy loading
     */
    public function optimize(string $html): string {
        $excluded = $this->nitro->get_setting('lazyload_exclude', []);

        // Lazy load images
        if ($this->nitro->get_setting('lazyload_images')) {
            $html = $this->lazyload_images($html, $excluded);
        }

        // Lazy load iframes
        if ($this->nitro->get_setting('lazyload_iframes')) {
            $html = $this->lazyload_iframes($html, $excluded);
        }

        // YouTube thumbnail replacement
        if ($this->nitro->get_setting('lazyload_youtube')) {
            $html = $this->replace_youtube_iframes($html);
        }

        return $html;
    }

    /**
     * Lazy load images
     */
    private function lazyload_images(string $html, array $excluded): string {
        // Match img tags
        $pattern = '/<img([^>]+)>/i';

        $html = preg_replace_callback($pattern, function ($matches) use ($excluded) {
            $tag = $matches[0];
            $attrs = $matches[1];

            // Skip if already has loading="lazy"
            if (strpos($attrs, 'loading="lazy"') !== false) {
                return $tag;
            }

            // Skip if in excluded list
            foreach ($excluded as $pattern) {
                if (empty($pattern)) continue;
                if (strpos($tag, $pattern) !== false) {
                    return $tag;
                }
            }

            // Skip images marked with data-no-lazy
            if (strpos($attrs, 'data-no-lazy') !== false) {
                return $tag;
            }

            // Skip first few images (likely above the fold)
            static $image_count = 0;
            $image_count++;
            if ($image_count <= 2) {
                // Add fetchpriority for first image
                if ($image_count === 1 && strpos($attrs, 'fetchpriority') === false) {
                    return '<img fetchpriority="high"' . $attrs . '>';
                }
                return $tag;
            }

            // Extract src
            if (!preg_match('/src=["\']([^"\']+)["\']/i', $attrs, $src_match)) {
                return $tag;
            }

            $src = $src_match[1];

            // Skip data URIs
            if (strpos($src, 'data:') === 0) {
                return $tag;
            }

            // Replace src with data-src
            $new_attrs = preg_replace('/src=["\']([^"\']+)["\']/i', 'src="' . self::PLACEHOLDER . '" data-src="$1"', $attrs);

            // Handle srcset
            if (preg_match('/srcset=["\']([^"\']+)["\']/i', $new_attrs)) {
                $new_attrs = preg_replace('/srcset=["\']([^"\']+)["\']/i', 'data-srcset="$1"', $new_attrs);
            }

            // Handle sizes
            if (preg_match('/sizes=["\']([^"\']+)["\']/i', $new_attrs)) {
                $new_attrs = preg_replace('/sizes=["\']([^"\']+)["\']/i', 'data-sizes="$1"', $new_attrs);
            }

            // Add lazy class
            if (preg_match('/class=["\']([^"\']+)["\']/i', $new_attrs, $class_match)) {
                $new_attrs = preg_replace('/class=["\']([^"\']+)["\']/i', 'class="$1 enbombas-lazy"', $new_attrs);
            } else {
                $new_attrs .= ' class="enbombas-lazy"';
            }

            return '<img' . $new_attrs . '>';
        }, $html);

        return $html;
    }

    /**
     * Lazy load iframes
     */
    private function lazyload_iframes(string $html, array $excluded): string {
        // Skip if YouTube replacement is enabled (it handles its own iframes)
        if ($this->nitro->get_setting('lazyload_youtube')) {
            // Only process non-YouTube iframes
            $pattern = '/<iframe([^>]+)>/i';
        } else {
            $pattern = '/<iframe([^>]+)>/i';
        }

        $html = preg_replace_callback($pattern, function ($matches) use ($excluded) {
            $tag = $matches[0];
            $attrs = $matches[1];

            // Skip if already has loading="lazy"
            if (strpos($attrs, 'loading="lazy"') !== false) {
                return $tag;
            }

            // Skip if in excluded list
            foreach ($excluded as $pattern) {
                if (empty($pattern)) continue;
                if (strpos($tag, $pattern) !== false) {
                    return $tag;
                }
            }

            // Skip if marked with data-no-lazy
            if (strpos($attrs, 'data-no-lazy') !== false) {
                return $tag;
            }

            // Skip YouTube if YouTube replacement is enabled
            if ($this->nitro->get_setting('lazyload_youtube') && 
                (strpos($attrs, 'youtube.com') !== false || strpos($attrs, 'youtu.be') !== false)) {
                return $tag;
            }

            // Extract src
            if (!preg_match('/src=["\']([^"\']+)["\']/i', $attrs, $src_match)) {
                return $tag;
            }

            $src = $src_match[1];

            // Replace src with data-src
            $new_attrs = preg_replace('/src=["\']([^"\']+)["\']/i', 'data-src="$1"', $attrs);

            // Add lazy class
            if (preg_match('/class=["\']([^"\']+)["\']/i', $new_attrs, $class_match)) {
                $new_attrs = preg_replace('/class=["\']([^"\']+)["\']/i', 'class="$1 enbombas-lazy-iframe"', $new_attrs);
            } else {
                $new_attrs .= ' class="enbombas-lazy-iframe"';
            }

            return '<iframe' . $new_attrs . '>';
        }, $html);

        return $html;
    }

    /**
     * Replace YouTube iframes with thumbnails
     */
    private function replace_youtube_iframes(string $html): string {
        // Match YouTube iframes
        $pattern = '/<iframe[^>]+(?:youtube\.com\/embed\/|youtu\.be\/)([a-zA-Z0-9_-]{11})[^>]*><\/iframe>/i';

        $html = preg_replace_callback($pattern, function ($matches) {
            $video_id = $matches[1];
            $original_iframe = $matches[0];

            // Extract dimensions
            $width = '100%';
            $height = '315';

            if (preg_match('/width=["\'](\d+)["\']/i', $original_iframe, $w_match)) {
                $width = $w_match[1] . 'px';
            }

            if (preg_match('/height=["\'](\d+)["\']/i', $original_iframe, $h_match)) {
                $height = $h_match[1] . 'px';
            }

            // Get thumbnail URL (maxresdefault or hqdefault)
            $thumbnail_url = "https://i.ytimg.com/vi/{$video_id}/hqdefault.jpg";

            // Build replacement HTML
            $replacement = <<<HTML
<div class="enbombas-youtube-facade" data-video-id="{$video_id}" style="width:{$width};height:{$height};">
    <img src="{$thumbnail_url}" alt="YouTube video" loading="lazy" style="width:100%;height:100%;object-fit:cover;">
    <button class="enbombas-youtube-play" aria-label="Play Video">
        <svg viewBox="0 0 68 48" width="68" height="48">
            <path class="enbombas-youtube-bg" d="M66.52,7.74c-0.78-2.93-2.49-5.41-5.42-6.19C55.79,.13,34,0,34,0S12.21,.13,6.9,1.55 C3.97,2.33,2.27,4.81,1.48,7.74C0.06,13.05,0,24,0,24s0.06,10.95,1.48,16.26c0.78,2.93,2.49,5.41,5.42,6.19 C12.21,47.87,34,48,34,48s21.79-0.13,27.1-1.55c2.93-0.78,4.64-3.26,5.42-6.19C67.94,34.95,68,24,68,24S67.94,13.05,66.52,7.74z" fill="#212121" fill-opacity="0.8"/>
            <path d="M 45,24 27,14 27,34" fill="#fff"/>
        </svg>
    </button>
</div>
HTML;

            return $replacement;
        }, $html);

        // Add YouTube facade styles and script if we have replacements
        if (strpos($html, 'enbombas-youtube-facade') !== false) {
            $styles = <<<CSS
<style>
.enbombas-youtube-facade{position:relative;cursor:pointer;background:#000;overflow:hidden}
.enbombas-youtube-facade img{transition:opacity 0.3s}
.enbombas-youtube-facade:hover img{opacity:0.7}
.enbombas-youtube-play{position:absolute;top:50%;left:50%;transform:translate(-50%,-50%);border:0;background:0 0;cursor:pointer;padding:0}
.enbombas-youtube-play:hover .enbombas-youtube-bg{fill:#f00;fill-opacity:1}
.enbombas-youtube-facade iframe{position:absolute;top:0;left:0;width:100%;height:100%}
</style>
CSS;

            $script = <<<JS
<script>
document.addEventListener('click',function(e){
    var facade = e.target.closest('.enbombas-youtube-facade');
    if(!facade) return;
    var id = facade.dataset.videoId;
    if(!id) return;
    var iframe = document.createElement('iframe');
    iframe.src = 'https://www.youtube.com/embed/' + id + '?autoplay=1&rel=0';
    iframe.allow = 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture';
    iframe.allowFullscreen = true;
    iframe.frameBorder = '0';
    facade.innerHTML = '';
    facade.appendChild(iframe);
});
</script>
JS;

            // Inject before </head> and before </body>
            $html = str_replace('</head>', $styles . "\n</head>", $html);
            $html = str_replace('</body>', $script . "\n</body>", $html);
        }

        return $html;
    }

    /**
     * Get inline styles for lazy loading
     */
    public function get_inline_styles(): string {
        return <<<CSS
.enbombas-lazy,
.enbombas-lazy-iframe {
    opacity: 0;
    transition: opacity 0.3s ease-in-out;
}
.enbombas-lazy.enbombas-loaded,
.enbombas-lazy-iframe.enbombas-loaded {
    opacity: 1;
}
CSS;
    }
}



