<?php
/**
 * JS Optimizer
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class JSOptimizer
 *
 * Handles JS minification, combining, defer, and delay.
 */
class JSOptimizer {
    /**
     * Nitro core instance
     */
    private NitroCore $nitro;

    /**
     * Cache directory
     */
    private string $cache_dir;

    /**
     * Cache URL
     */
    private string $cache_url;

    /**
     * Scripts to delay
     */
    private array $delayed_scripts = [];

    /**
     * Constructor
     */
    public function __construct(NitroCore $nitro) {
        $this->nitro = $nitro;

        $upload_dir = wp_upload_dir();
        $this->cache_dir = $upload_dir['basedir'] . '/enbombas-nitro/js/';
        $this->cache_url = $upload_dir['baseurl'] . '/enbombas-nitro/js/';

        $this->ensure_cache_dir();
    }

    /**
     * Ensure cache directory exists
     */
    private function ensure_cache_dir(): void {
        if (!file_exists($this->cache_dir)) {
            wp_mkdir_p($this->cache_dir);
            file_put_contents($this->cache_dir . 'index.php', '<?php // Silence');
        }
    }

    /**
     * Optimize JS in HTML
     */
    public function optimize(string $html): string {
        // Extract all script tags
        $pattern = '/<script[^>]*>[\s\S]*?<\/script>/i';

        if (!preg_match_all($pattern, $html, $matches)) {
            return $html;
        }

        $script_tags = $matches[0];
        $excluded = $this->nitro->get_setting('js_exclude', []);

        $to_process = [];
        $inline_scripts = [];

        foreach ($script_tags as $tag) {
            if ($this->should_exclude($tag, $excluded)) {
                continue;
            }

            // Check if it's an external script
            if (preg_match('/src=["\']([^"\']+)["\']/i', $tag, $src_match)) {
                $to_process[] = [
                    'tag' => $tag,
                    'src' => $src_match[1],
                    'type' => 'external',
                ];
            } else {
                // Inline script
                if (preg_match('/<script[^>]*>([\s\S]*?)<\/script>/i', $tag, $content_match)) {
                    $inline_scripts[] = [
                        'tag' => $tag,
                        'content' => $content_match[1],
                        'type' => 'inline',
                    ];
                }
            }
        }

        // Combine JS if enabled
        if ($this->nitro->get_setting('js_combine') && count($to_process) > 1) {
            $combined = $this->combine_js($to_process);
            if ($combined) {
                // Remove all original tags
                foreach ($to_process as $item) {
                    $html = str_replace($item['tag'], '', $html);
                }
                // Add combined JS
                $combined_tag = $this->create_script_tag($combined['url']);
                $html = str_replace('</body>', $combined_tag . "\n</body>", $html);

                // Process remaining inline scripts
                foreach ($inline_scripts as $item) {
                    $new_tag = $this->process_inline_script($item);
                    $html = str_replace($item['tag'], $new_tag, $html);
                }

                // Add delay script if needed
                if ($this->nitro->get_setting('js_delay')) {
                    $html = $this->add_delay_loader($html);
                }

                return $html;
            }
        }

        // Process individual scripts
        foreach ($to_process as $item) {
            $new_tag = $this->process_script_tag($item['tag'], $item['src']);
            $html = str_replace($item['tag'], $new_tag, $html);
        }

        // Process inline scripts
        foreach ($inline_scripts as $item) {
            if ($this->nitro->get_setting('js_minify')) {
                $new_tag = $this->process_inline_script($item);
                $html = str_replace($item['tag'], $new_tag, $html);
            }
        }

        // Add delay loader if enabled
        if ($this->nitro->get_setting('js_delay') && !empty($this->delayed_scripts)) {
            $html = $this->add_delay_loader($html);
        }

        return $html;
    }

    /**
     * Process individual script tag
     */
    private function process_script_tag(string $tag, string $src): string {
        $url = $src;

        // Minify if enabled and local file
        if ($this->nitro->get_setting('js_minify')) {
            $minified = $this->minify_file($src);
            if ($minified) {
                $url = $minified;
            }
        }

        // Delay JS execution
        if ($this->nitro->get_setting('js_delay')) {
            $this->delayed_scripts[] = $url;
            return '<!-- Delayed: ' . esc_html($src) . ' -->';
        }

        // Add defer attribute
        if ($this->nitro->get_setting('js_defer')) {
            // Remove existing async/defer
            $tag = preg_replace('/\s+(async|defer)(\s+|=)/i', ' ', $tag);
            
            // Add defer
            $tag = str_replace('<script', '<script defer', $tag);

            // Update URL if minified
            if ($url !== $src) {
                $tag = str_replace($src, $url, $tag);
            }

            return $tag;
        }

        // Just update URL if minified
        if ($url !== $src) {
            return str_replace($src, $url, $tag);
        }

        return $tag;
    }

    /**
     * Process inline script
     */
    private function process_inline_script(array $item): string {
        $content = $item['content'];

        // Minify inline script
        if ($this->nitro->get_setting('js_minify') && !empty(trim($content))) {
            $content = $this->minify_js($content);
        }

        // Check if should delay
        if ($this->nitro->get_setting('js_delay') && !$this->is_critical_inline($content)) {
            // Convert to data-src for delay
            return '<script type="text/enbombas-delay">' . $content . '</script>';
        }

        return '<script>' . $content . '</script>';
    }

    /**
     * Check if inline script is critical (should not be delayed)
     */
    private function is_critical_inline(string $content): bool {
        $critical_patterns = [
            'document.write',
            'document.documentElement',
            'var wpml',
            'var wc_',
            'window.dataLayer',
            'gtag(',
            'fbq(',
        ];

        foreach ($critical_patterns as $pattern) {
            if (stripos($content, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Combine multiple JS files
     */
    private function combine_js(array $items): ?array {
        $combined_content = '';
        $hash_parts = [];

        foreach ($items as $item) {
            $content = $this->get_js_content($item['src']);
            if ($content) {
                $combined_content .= "\n/* Source: {$item['src']} */\n" . $content . ";\n";
                $hash_parts[] = md5($item['src']);
            }
        }

        if (empty($combined_content)) {
            return null;
        }

        // Minify if enabled
        if ($this->nitro->get_setting('js_minify')) {
            $combined_content = $this->minify_js($combined_content);
        }

        // Generate filename
        $hash = md5(implode('', $hash_parts));
        $filename = 'combined-' . $hash . '.js';
        $filepath = $this->cache_dir . $filename;

        // Save combined file
        file_put_contents($filepath, $combined_content);

        return [
            'url' => $this->cache_url . $filename,
            'hash' => $hash,
        ];
    }

    /**
     * Minify a JS file
     */
    private function minify_file(string $url): ?string {
        $content = $this->get_js_content($url);
        if (!$content) {
            return null;
        }

        // Minify
        $minified = $this->minify_js($content);

        // Generate filename
        $hash = md5($url . $minified);
        $filename = 'min-' . $hash . '.js';
        $filepath = $this->cache_dir . $filename;

        // Check if already cached
        if (!file_exists($filepath)) {
            file_put_contents($filepath, $minified);
        }

        return $this->cache_url . $filename;
    }

    /**
     * Minify JS content
     */
    private function minify_js(string $js): string {
        // Don't minify if already minified (simple check)
        if (strlen($js) > 0 && substr_count($js, "\n") < strlen($js) / 500) {
            return $js;
        }

        // Remove single-line comments (but not URLs)
        $js = preg_replace('/(?<!:)\/\/[^\n]*$/m', '', $js);

        // Remove multi-line comments
        $js = preg_replace('/\/\*[\s\S]*?\*\//', '', $js);

        // Remove extra whitespace
        $js = preg_replace('/\s+/', ' ', $js);

        // Remove spaces around operators (basic)
        $js = preg_replace('/\s*([=+\-*\/%<>&|!?:,;{}()\[\]])\s*/', '$1', $js);

        // Restore spaces where needed
        $js = preg_replace('/([a-z0-9_$])(var|let|const|function|return|if|else|for|while|do|switch|case|break|continue|new|delete|typeof|instanceof|in|of|class|extends|import|export|default|from|as|try|catch|finally|throw|async|await)\b/i', '$1 $2', $js);
        $js = preg_replace('/\b(var|let|const|function|return|if|else|for|while|do|switch|case|break|continue|new|delete|typeof|instanceof|in|of|class|extends|import|export|default|from|as|try|catch|finally|throw|async|await)([a-z0-9_$])/i', '$1 $2', $js);

        // Remove newlines
        $js = str_replace(["\r\n", "\r", "\n"], '', $js);

        return trim($js);
    }

    /**
     * Get JS content from URL
     */
    private function get_js_content(string $url): ?string {
        // Convert URL to path for local files
        $path = $this->url_to_path($url);

        if ($path && file_exists($path)) {
            return file_get_contents($path);
        }

        // Try to fetch remote JS
        $response = wp_remote_get($url, [
            'timeout' => 10,
            'sslverify' => false,
        ]);

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            return wp_remote_retrieve_body($response);
        }

        return null;
    }

    /**
     * Convert URL to local file path
     */
    private function url_to_path(string $url): ?string {
        $site_url = site_url();
        $home_url = home_url();

        // Try site_url
        if (strpos($url, $site_url) === 0) {
            $path = ABSPATH . str_replace($site_url . '/', '', $url);
            if (file_exists($path)) {
                return $path;
            }
        }

        // Try home_url
        if (strpos($url, $home_url) === 0) {
            $path = ABSPATH . str_replace($home_url . '/', '', $url);
            if (file_exists($path)) {
                return $path;
            }
        }

        // Try relative URL
        if (strpos($url, '//') !== 0 && strpos($url, 'http') !== 0) {
            $path = ABSPATH . ltrim($url, '/');
            if (file_exists($path)) {
                return $path;
            }
        }

        return null;
    }

    /**
     * Create script tag
     */
    private function create_script_tag(string $url): string {
        $attrs = '';

        if ($this->nitro->get_setting('js_delay')) {
            // Will be loaded by delay loader
            return '<script type="text/enbombas-delay" data-src="' . esc_url($url) . '"></script>';
        }

        if ($this->nitro->get_setting('js_defer')) {
            $attrs = ' defer';
        }

        return '<script src="' . esc_url($url) . '"' . $attrs . '></script>';
    }

    /**
     * Add delay loader script
     */
    private function add_delay_loader(string $html): string {
        $timeout = $this->nitro->get_setting('js_delay_timeout', 5000);
        $scripts_json = json_encode($this->delayed_scripts);

        $loader = <<<JS
<script id="enbombas-delay-loader">
(function(){
    var loaded = false;
    var scripts = {$scripts_json};
    var timeout = {$timeout};
    
    function loadDelayed() {
        if (loaded) return;
        loaded = true;
        
        // Load external scripts
        scripts.forEach(function(src) {
            var s = document.createElement('script');
            s.src = src;
            s.async = true;
            document.body.appendChild(s);
        });
        
        // Execute inline delayed scripts
        document.querySelectorAll('script[type="text/enbombas-delay"]').forEach(function(el) {
            if (el.dataset.src) return; // External, already handled
            var s = document.createElement('script');
            s.textContent = el.textContent;
            document.body.appendChild(s);
            el.remove();
        });
    }
    
    // Trigger events
    var events = ['scroll', 'click', 'mousemove', 'keydown', 'touchstart'];
    events.forEach(function(e) {
        window.addEventListener(e, loadDelayed, {once: true, passive: true});
    });
    
    // Fallback timeout
    setTimeout(loadDelayed, timeout);
})();
</script>
JS;

        // Insert before </body>
        $html = str_replace('</body>', $loader . "\n</body>", $html);

        return $html;
    }

    /**
     * Check if tag should be excluded
     */
    private function should_exclude(string $tag, array $excluded): bool {
        // Always exclude jQuery core and WordPress core scripts
        $always_exclude = [
            'jquery.min.js',
            'jquery.js',
            'wp-includes/js/dist/',
            'wp-includes/js/tinymce/',
            'admin-bar',
            'enbombas-nitro',
        ];

        foreach ($always_exclude as $pattern) {
            if (strpos($tag, $pattern) !== false) {
                return true;
            }
        }

        foreach ($excluded as $pattern) {
            if (empty($pattern)) continue;

            if (strpos($tag, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Clear JS cache
     */
    public function clear_cache(): int {
        $count = 0;

        if (is_dir($this->cache_dir)) {
            $files = glob($this->cache_dir . '*.js');
            foreach ($files as $file) {
                if (unlink($file)) {
                    $count++;
                }
            }
        }

        return $count;
    }
}



