<?php
/**
 * Heartbeat Control
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class Heartbeat
 *
 * Controls WordPress Heartbeat API settings.
 */
class Heartbeat {
    /**
     * Nitro core instance
     */
    private NitroCore $nitro;

    /**
     * Constructor
     */
    public function __construct(NitroCore $nitro) {
        $this->nitro = $nitro;
        $this->register_hooks();
    }

    /**
     * Register hooks
     */
    private function register_hooks(): void {
        // Modify heartbeat settings
        add_filter('heartbeat_settings', [$this, 'modify_heartbeat_settings'], 10, 1);

        // Disable heartbeat completely if needed
        add_action('init', [$this, 'maybe_disable_heartbeat'], 1);

        // Backend-only mode
        add_action('wp_enqueue_scripts', [$this, 'maybe_dequeue_heartbeat'], 100);
    }

    /**
     * Modify heartbeat settings
     */
    public function modify_heartbeat_settings(array $settings): array {
        $behavior = $this->nitro->get_setting('heartbeat_behavior', 'default');

        if ($behavior === 'reduce') {
            $frequency = $this->nitro->get_setting('heartbeat_frequency', 60);
            $settings['interval'] = max(15, min(120, $frequency)); // Clamp between 15-120 seconds
        }

        return $settings;
    }

    /**
     * Maybe disable heartbeat completely
     */
    public function maybe_disable_heartbeat(): void {
        $behavior = $this->nitro->get_setting('heartbeat_behavior', 'default');

        if ($behavior === 'disable') {
            wp_deregister_script('heartbeat');
        }
    }

    /**
     * Maybe dequeue heartbeat on frontend
     */
    public function maybe_dequeue_heartbeat(): void {
        if ($this->nitro->get_setting('heartbeat_backend_only')) {
            wp_deregister_script('heartbeat');
        }
    }

    /**
     * Get heartbeat status
     */
    public function get_status(): array {
        $behavior = $this->nitro->get_setting('heartbeat_behavior', 'default');
        $frequency = $this->nitro->get_setting('heartbeat_frequency', 60);
        $backend_only = $this->nitro->get_setting('heartbeat_backend_only', false);

        $status = [
            'behavior' => $behavior,
            'frequency' => $frequency,
            'backend_only' => $backend_only,
            'status_text' => '',
            'status_class' => '',
        ];

        switch ($behavior) {
            case 'disable':
                $status['status_text'] = __('Deshabilitado completamente', 'en-bombas');
                $status['status_class'] = 'status-disabled';
                break;

            case 'reduce':
                $status['status_text'] = sprintf(
                    __('Reducido a cada %d segundos', 'en-bombas'),
                    $frequency
                );
                $status['status_class'] = 'status-reduced';
                break;

            default:
                if ($backend_only) {
                    $status['status_text'] = __('Solo en el escritorio (admin)', 'en-bombas');
                    $status['status_class'] = 'status-backend';
                } else {
                    $status['status_text'] = __('Configuración por defecto de WordPress', 'en-bombas');
                    $status['status_class'] = 'status-default';
                }
                break;
        }

        return $status;
    }

    /**
     * Get heartbeat info text
     */
    public function get_info(): string {
        return __(
            'El Heartbeat de WordPress es una API que envía solicitudes AJAX periódicas para mantener ' .
            'funciones como autoguardado, bloqueo de posts y notificaciones en tiempo real. ' .
            'Reducir o desactivar el Heartbeat puede mejorar el rendimiento, especialmente en ' .
            'servidores compartidos, pero puede afectar algunas funcionalidades del editor.',
            'en-bombas'
        );
    }
}



