<?php
/**
 * Database Optimization
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class Database
 *
 * Handles database cleanup and optimization.
 */
class Database {
    /**
     * Nitro core instance
     */
    private NitroCore $nitro;

    /**
     * Constructor
     */
    public function __construct(NitroCore $nitro) {
        $this->nitro = $nitro;
    }

    /**
     * Run database cleanup
     */
    public function run_cleanup(): array {
        global $wpdb;

        $results = [
            'revisions' => 0,
            'drafts' => 0,
            'trash' => 0,
            'transients' => 0,
            'spam_comments' => 0,
            'trash_comments' => 0,
            'orphan_meta' => 0,
            'tables_optimized' => 0,
        ];

        // Clean revisions
        if ($this->nitro->get_setting('db_clean_revisions')) {
            $results['revisions'] = $this->clean_revisions();
        }

        // Clean auto-drafts
        if ($this->nitro->get_setting('db_clean_drafts')) {
            $results['drafts'] = $this->clean_auto_drafts();
        }

        // Clean trash
        if ($this->nitro->get_setting('db_clean_trash')) {
            $results['trash'] = $this->clean_trash();
            $results['spam_comments'] = $this->clean_spam_comments();
            $results['trash_comments'] = $this->clean_trash_comments();
        }

        // Clean transients
        if ($this->nitro->get_setting('db_clean_transients')) {
            $results['transients'] = $this->clean_transients();
        }

        // Clean orphan meta
        $results['orphan_meta'] = $this->clean_orphan_meta();

        // Optimize tables
        if ($this->nitro->get_setting('db_optimize_tables')) {
            $results['tables_optimized'] = $this->optimize_tables();
        }

        return $results;
    }

    /**
     * Clean post revisions
     */
    public function clean_revisions(): int {
        global $wpdb;

        $count = $wpdb->query(
            "DELETE FROM {$wpdb->posts} WHERE post_type = 'revision'"
        );

        return $count !== false ? $count : 0;
    }

    /**
     * Clean auto-drafts
     */
    public function clean_auto_drafts(): int {
        global $wpdb;

        $count = $wpdb->query(
            "DELETE FROM {$wpdb->posts} WHERE post_status = 'auto-draft'"
        );

        return $count !== false ? $count : 0;
    }

    /**
     * Clean trashed posts
     */
    public function clean_trash(): int {
        global $wpdb;

        // Get trashed post IDs first
        $trashed_ids = $wpdb->get_col(
            "SELECT ID FROM {$wpdb->posts} WHERE post_status = 'trash'"
        );

        if (empty($trashed_ids)) {
            return 0;
        }

        // Delete post meta for trashed posts
        $ids_placeholder = implode(',', array_map('absint', $trashed_ids));
        $wpdb->query(
            "DELETE FROM {$wpdb->postmeta} WHERE post_id IN ({$ids_placeholder})"
        );

        // Delete the trashed posts
        $count = $wpdb->query(
            "DELETE FROM {$wpdb->posts} WHERE post_status = 'trash'"
        );

        return $count !== false ? $count : 0;
    }

    /**
     * Clean spam comments
     */
    public function clean_spam_comments(): int {
        global $wpdb;

        $count = $wpdb->query(
            "DELETE FROM {$wpdb->comments} WHERE comment_approved = 'spam'"
        );

        // Clean orphan comment meta
        $wpdb->query(
            "DELETE FROM {$wpdb->commentmeta} WHERE comment_id NOT IN (SELECT comment_ID FROM {$wpdb->comments})"
        );

        return $count !== false ? $count : 0;
    }

    /**
     * Clean trashed comments
     */
    public function clean_trash_comments(): int {
        global $wpdb;

        $count = $wpdb->query(
            "DELETE FROM {$wpdb->comments} WHERE comment_approved = 'trash'"
        );

        return $count !== false ? $count : 0;
    }

    /**
     * Clean expired transients
     */
    public function clean_transients(): int {
        global $wpdb;

        $time = time();
        $count = 0;

        // Delete expired transients
        $expired = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} 
                WHERE option_name LIKE %s 
                AND option_value < %d",
                $wpdb->esc_like('_transient_timeout_') . '%',
                $time
            )
        );

        if (!empty($expired)) {
            foreach ($expired as $transient) {
                $name = str_replace('_transient_timeout_', '', $transient);
                delete_transient($name);
                $count++;
            }
        }

        // Delete orphan transient timeouts
        $wpdb->query(
            "DELETE FROM {$wpdb->options} 
            WHERE option_name LIKE '_transient_timeout_%' 
            AND option_name NOT IN (
                SELECT CONCAT('_transient_timeout_', SUBSTRING(option_name, 12)) 
                FROM (SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE '_transient_%') AS t
            )"
        );

        return $count;
    }

    /**
     * Clean orphan post meta
     */
    public function clean_orphan_meta(): int {
        global $wpdb;

        $count = $wpdb->query(
            "DELETE pm FROM {$wpdb->postmeta} pm 
            LEFT JOIN {$wpdb->posts} p ON p.ID = pm.post_id 
            WHERE p.ID IS NULL"
        );

        return $count !== false ? $count : 0;
    }

    /**
     * Optimize database tables
     */
    public function optimize_tables(): int {
        global $wpdb;

        $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);
        $count = 0;

        foreach ($tables as $table) {
            $table_name = $table[0];
            
            // Only optimize WordPress tables
            if (strpos($table_name, $wpdb->prefix) !== 0) {
                continue;
            }

            $result = $wpdb->query("OPTIMIZE TABLE `{$table_name}`");
            if ($result !== false) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Get database statistics
     */
    public function get_stats(): array {
        global $wpdb;

        $stats = [
            'revisions' => 0,
            'drafts' => 0,
            'trash' => 0,
            'spam_comments' => 0,
            'trash_comments' => 0,
            'transients' => 0,
            'orphan_meta' => 0,
            'db_size' => 0,
            'db_size_formatted' => '0 B',
            'tables' => [],
        ];

        // Count revisions
        $stats['revisions'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'revision'"
        );

        // Count auto-drafts
        $stats['drafts'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'auto-draft'"
        );

        // Count trashed posts
        $stats['trash'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'trash'"
        );

        // Count spam comments
        $stats['spam_comments'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = 'spam'"
        );

        // Count trashed comments
        $stats['trash_comments'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = 'trash'"
        );

        // Count expired transients
        $stats['transients'] = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->options} 
                WHERE option_name LIKE %s 
                AND option_value < %d",
                $wpdb->esc_like('_transient_timeout_') . '%',
                time()
            )
        );

        // Count orphan post meta
        $stats['orphan_meta'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm 
            LEFT JOIN {$wpdb->posts} p ON p.ID = pm.post_id 
            WHERE p.ID IS NULL"
        );

        // Get database size
        $tables = $wpdb->get_results(
            "SELECT table_name AS name, 
                    ROUND((data_length + index_length), 0) AS size,
                    ROUND((data_free), 0) AS overhead
            FROM information_schema.tables 
            WHERE table_schema = '" . DB_NAME . "' 
            AND table_name LIKE '{$wpdb->prefix}%'
            ORDER BY (data_length + index_length) DESC"
        );

        $total_size = 0;
        $total_overhead = 0;

        foreach ($tables as $table) {
            $total_size += $table->size;
            $total_overhead += $table->overhead;

            $stats['tables'][] = [
                'name' => $table->name,
                'size' => size_format($table->size),
                'overhead' => size_format($table->overhead),
            ];
        }

        $stats['db_size'] = $total_size;
        $stats['db_size_formatted'] = size_format($total_size);
        $stats['db_overhead'] = $total_overhead;
        $stats['db_overhead_formatted'] = size_format($total_overhead);

        // Calculate potential savings
        $stats['potential_savings'] = $stats['revisions'] + $stats['drafts'] + $stats['trash'] + 
                                      $stats['spam_comments'] + $stats['trash_comments'] + 
                                      $stats['transients'] + $stats['orphan_meta'];

        return $stats;
    }

    /**
     * Schedule automatic cleanup
     */
    public function schedule_cleanup(): void {
        $frequency = $this->nitro->get_setting('db_auto_clean_frequency', 'weekly');

        if (!wp_next_scheduled('enbombas_nitro_db_cleanup')) {
            wp_schedule_event(time(), $frequency, 'enbombas_nitro_db_cleanup');
        }
    }

    /**
     * Unschedule automatic cleanup
     */
    public function unschedule_cleanup(): void {
        $timestamp = wp_next_scheduled('enbombas_nitro_db_cleanup');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'enbombas_nitro_db_cleanup');
        }
    }

    /**
     * Get cleanup schedule info
     */
    public function get_schedule_info(): array {
        $next = wp_next_scheduled('enbombas_nitro_db_cleanup');

        return [
            'scheduled' => $next !== false,
            'next_run' => $next ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $next) : null,
            'frequency' => $this->nitro->get_setting('db_auto_clean_frequency', 'weekly'),
        ];
    }
}



