<?php
/**
 * Cache System
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class CacheSystem
 *
 * Handles HTML page caching.
 */
class CacheSystem {
    /**
     * Nitro core instance
     */
    private NitroCore $nitro;

    /**
     * Cache directory
     */
    private string $cache_dir;

    /**
     * Constructor
     */
    public function __construct(NitroCore $nitro) {
        $this->nitro = $nitro;
        $this->cache_dir = WP_CONTENT_DIR . '/cache/en-bombas/';
        $this->ensure_cache_dir();
    }

    /**
     * Ensure cache directory exists
     */
    private function ensure_cache_dir(): void {
        if (!file_exists($this->cache_dir)) {
            wp_mkdir_p($this->cache_dir);
            
            // Add index.php for security
            file_put_contents($this->cache_dir . 'index.php', '<?php // Silence is golden');
            
            // Add .htaccess to serve gzipped files
            $htaccess = <<<HTACCESS
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteCond %{HTTP:Accept-Encoding} gzip
    RewriteCond %{REQUEST_FILENAME}.gz -f
    RewriteRule ^(.*)$ $1.gz [L]
</IfModule>

<IfModule mod_headers.c>
    <FilesMatch "\.html\.gz$">
        Header set Content-Encoding gzip
        Header set Content-Type "text/html; charset=UTF-8"
    </FilesMatch>
</IfModule>
HTACCESS;
            file_put_contents($this->cache_dir . '.htaccess', $htaccess);
        }
    }

    /**
     * Get cache file path for current request
     */
    private function get_cache_path(): string {
        $url = $this->get_current_url();
        $hash = md5($url);
        
        // Create subdirectory based on hash prefix
        $subdir = substr($hash, 0, 2);
        $dir = $this->cache_dir . $subdir . '/';
        
        if (!file_exists($dir)) {
            wp_mkdir_p($dir);
        }

        // Add mobile suffix if needed
        $suffix = '';
        if ($this->nitro->get_setting('cache_mobile') && wp_is_mobile()) {
            $suffix = '-mobile';
        }

        return $dir . $hash . $suffix . '.html';
    }

    /**
     * Get current URL
     */
    private function get_current_url(): string {
        $scheme = is_ssl() ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $uri = $_SERVER['REQUEST_URI'] ?? '/';

        // Remove query strings that shouldn't affect cache
        $uri = preg_replace('/[?&](utm_[^&]*|fbclid|gclid|ref)=[^&]*/i', '', $uri);
        $uri = rtrim($uri, '?&');

        return $scheme . '://' . $host . $uri;
    }

    /**
     * Check if request should be cached
     */
    private function should_cache(): bool {
        // Don't cache POST requests
        if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
            return false;
        }

        // Don't cache admin, cron, AJAX
        if (is_admin() || wp_doing_cron() || wp_doing_ajax()) {
            return false;
        }

        // Don't cache preview
        if (is_preview()) {
            return false;
        }

        // Don't cache feed
        if (is_feed()) {
            return false;
        }

        // Don't cache search
        if (is_search()) {
            return false;
        }

        // Don't cache 404
        if (is_404()) {
            return false;
        }

        // Check excluded URLs
        $excluded_urls = $this->nitro->get_setting('cache_exclude_urls', []);
        $current_url = $this->get_current_url();

        foreach ($excluded_urls as $pattern) {
            if (empty($pattern)) continue;
            
            if (strpos($current_url, $pattern) !== false) {
                return false;
            }
            
            // Support wildcards
            $regex = str_replace('*', '.*', preg_quote($pattern, '/'));
            if (preg_match('/' . $regex . '/i', $current_url)) {
                return false;
            }
        }

        // Check excluded cookies
        $excluded_cookies = $this->nitro->get_setting('cache_exclude_cookies', []);
        foreach ($excluded_cookies as $cookie) {
            if (empty($cookie)) continue;
            
            if (isset($_COOKIE[$cookie])) {
                return false;
            }
        }

        // Don't cache if user is logged in (unless enabled)
        if (is_user_logged_in() && !$this->nitro->get_setting('cache_logged_in')) {
            return false;
        }

        // Don't cache WooCommerce cart/checkout
        if (function_exists('is_cart') && (is_cart() || is_checkout() || is_account_page())) {
            return false;
        }

        return true;
    }

    /**
     * Serve cached page if available
     */
    public function serve_cache(): bool {
        if (!$this->nitro->get_setting('cache_enabled')) {
            return false;
        }

        if (!$this->should_cache()) {
            return false;
        }

        $cache_path = $this->get_cache_path();

        if (!file_exists($cache_path)) {
            return false;
        }

        // Check cache lifespan
        $lifespan = $this->nitro->get_setting('cache_lifespan', 604800);
        $file_age = time() - filemtime($cache_path);

        if ($file_age > $lifespan) {
            unlink($cache_path);
            return false;
        }

        // Serve cached content
        $content = file_get_contents($cache_path);

        // Add cache header
        header('X-EnBombas-Cache: HIT');
        header('X-EnBombas-Cache-Age: ' . $file_age);

        echo $content;
        echo "\n<!-- Cached by En Bombas Nitro | " . date('Y-m-d H:i:s', filemtime($cache_path)) . " -->";

        return true;
    }

    /**
     * Save HTML to cache
     */
    public function save_cache(string $html): bool {
        if (!$this->nitro->get_setting('cache_enabled')) {
            return false;
        }

        if (!$this->should_cache()) {
            return false;
        }

        $cache_path = $this->get_cache_path();

        // Add cache timestamp comment
        $timestamp = date('Y-m-d H:i:s');
        $html .= "\n<!-- Cached by En Bombas Nitro | Generated: {$timestamp} -->";

        // Save HTML
        $result = file_put_contents($cache_path, $html);

        // Create gzipped version
        if ($result && function_exists('gzencode')) {
            $gzipped = gzencode($html, 9);
            file_put_contents($cache_path . '.gz', $gzipped);
        }

        return $result !== false;
    }

    /**
     * Clear cache for specific post
     */
    public function clear_post_cache(int $post_id): void {
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        // Clear post URL cache
        $url = get_permalink($post_id);
        if ($url) {
            $this->clear_url_cache($url);
        }

        // Clear home page cache
        $this->clear_url_cache(home_url('/'));

        // Clear archive pages
        $post_type = get_post_type($post_id);
        $archive_url = get_post_type_archive_link($post_type);
        if ($archive_url) {
            $this->clear_url_cache($archive_url);
        }

        // Clear category/tag archives
        $taxonomies = get_object_taxonomies($post_type);
        foreach ($taxonomies as $taxonomy) {
            $terms = get_the_terms($post_id, $taxonomy);
            if ($terms && !is_wp_error($terms)) {
                foreach ($terms as $term) {
                    $term_url = get_term_link($term);
                    if (!is_wp_error($term_url)) {
                        $this->clear_url_cache($term_url);
                    }
                }
            }
        }
    }

    /**
     * Clear cache for specific URL
     */
    public function clear_url_cache(string $url): bool {
        $hash = md5($url);
        $subdir = substr($hash, 0, 2);

        $paths = [
            $this->cache_dir . $subdir . '/' . $hash . '.html',
            $this->cache_dir . $subdir . '/' . $hash . '-mobile.html',
            $this->cache_dir . $subdir . '/' . $hash . '.html.gz',
            $this->cache_dir . $subdir . '/' . $hash . '-mobile.html.gz',
        ];

        $cleared = false;
        foreach ($paths as $path) {
            if (file_exists($path)) {
                unlink($path);
                $cleared = true;
            }
        }

        return $cleared;
    }

    /**
     * Clear all cache
     */
    public function clear_all_cache(): array {
        $result = [
            'success' => true,
            'files_deleted' => 0,
            'space_freed' => 0,
        ];

        if (!is_dir($this->cache_dir)) {
            return $result;
        }

        // Clear cache directory recursively
        $result = $this->delete_directory_contents($this->cache_dir, $result);

        // Recreate base files
        file_put_contents($this->cache_dir . 'index.php', '<?php // Silence is golden');

        return $result;
    }

    /**
     * Delete directory contents recursively
     */
    private function delete_directory_contents(string $dir, array $result): array {
        if (!is_dir($dir)) {
            return $result;
        }

        $items = scandir($dir);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }

            $path = $dir . $item;

            if (is_dir($path)) {
                $result = $this->delete_directory_contents($path . '/', $result);
                rmdir($path);
            } else {
                // Skip index.php and .htaccess
                if (in_array($item, ['index.php', '.htaccess'])) {
                    continue;
                }

                $size = filesize($path);
                if (unlink($path)) {
                    $result['files_deleted']++;
                    $result['space_freed'] += $size;
                }
            }
        }

        return $result;
    }

    /**
     * Get cache statistics
     */
    public function get_stats(): array {
        $stats = [
            'total_files' => 0,
            'total_size' => 0,
            'total_size_formatted' => '0 B',
            'oldest_cache' => null,
            'newest_cache' => null,
        ];

        if (!is_dir($this->cache_dir)) {
            return $stats;
        }

        $oldest = PHP_INT_MAX;
        $newest = 0;

        $this->scan_directory_stats($this->cache_dir, $stats, $oldest, $newest);

        if ($oldest !== PHP_INT_MAX) {
            $stats['oldest_cache'] = date('Y-m-d H:i:s', $oldest);
        }

        if ($newest !== 0) {
            $stats['newest_cache'] = date('Y-m-d H:i:s', $newest);
        }

        $stats['total_size_formatted'] = size_format($stats['total_size']);

        return $stats;
    }

    /**
     * Scan directory for stats recursively
     */
    private function scan_directory_stats(string $dir, array &$stats, int &$oldest, int &$newest): void {
        if (!is_dir($dir)) {
            return;
        }

        $items = scandir($dir);
        foreach ($items as $item) {
            if ($item === '.' || $item === '..') {
                continue;
            }

            $path = $dir . $item;

            if (is_dir($path)) {
                $this->scan_directory_stats($path . '/', $stats, $oldest, $newest);
            } else {
                // Skip non-cache files
                if (!preg_match('/\.html(\.gz)?$/', $item)) {
                    continue;
                }

                $stats['total_files']++;
                $stats['total_size'] += filesize($path);

                $mtime = filemtime($path);
                if ($mtime < $oldest) {
                    $oldest = $mtime;
                }
                if ($mtime > $newest) {
                    $newest = $mtime;
                }
            }
        }
    }

    /**
     * Preload cache for URLs
     */
    public function preload_cache(array $urls): array {
        $results = [];

        foreach ($urls as $url) {
            $response = wp_remote_get($url, [
                'timeout' => 30,
                'sslverify' => false,
                'user-agent' => 'EnBombas-Nitro-Preloader',
            ]);

            $results[$url] = !is_wp_error($response);
        }

        return $results;
    }

    /**
     * Get sitemap URLs for preloading
     */
    public function get_sitemap_urls(int $limit = 100): array {
        $urls = [];

        // Get recent posts
        $posts = get_posts([
            'post_type' => 'post',
            'posts_per_page' => $limit,
            'post_status' => 'publish',
        ]);

        foreach ($posts as $post) {
            $urls[] = get_permalink($post);
        }

        // Get pages
        $pages = get_posts([
            'post_type' => 'page',
            'posts_per_page' => $limit,
            'post_status' => 'publish',
        ]);

        foreach ($pages as $page) {
            $urls[] = get_permalink($page);
        }

        // Add home
        array_unshift($urls, home_url('/'));

        return array_slice(array_unique($urls), 0, $limit);
    }
}



