<?php
/**
 * CSS Optimizer
 *
 * @package EnBombas\Nitro
 */

namespace EnBombas\Nitro;

/**
 * Class CSSOptimizer
 *
 * Handles CSS minification, combining, and async loading.
 */
class CSSOptimizer {
    /**
     * Nitro core instance
     */
    private NitroCore $nitro;

    /**
     * Cache directory for optimized CSS
     */
    private string $cache_dir;

    /**
     * Cache URL
     */
    private string $cache_url;

    /**
     * Constructor
     */
    public function __construct(NitroCore $nitro) {
        $this->nitro = $nitro;
        
        $upload_dir = wp_upload_dir();
        $this->cache_dir = $upload_dir['basedir'] . '/enbombas-nitro/css/';
        $this->cache_url = $upload_dir['baseurl'] . '/enbombas-nitro/css/';

        $this->ensure_cache_dir();
    }

    /**
     * Ensure cache directory exists
     */
    private function ensure_cache_dir(): void {
        if (!file_exists($this->cache_dir)) {
            wp_mkdir_p($this->cache_dir);
            file_put_contents($this->cache_dir . 'index.php', '<?php // Silence');
        }
    }

    /**
     * Optimize CSS in HTML
     */
    public function optimize(string $html): string {
        // Extract all CSS links
        $pattern = '/<link[^>]+rel=["\']stylesheet["\'][^>]*>/i';
        
        if (!preg_match_all($pattern, $html, $matches)) {
            return $html;
        }

        $css_tags = $matches[0];
        $excluded = $this->nitro->get_setting('css_exclude', []);
        
        $to_process = [];
        $to_keep = [];

        foreach ($css_tags as $tag) {
            if ($this->should_exclude($tag, $excluded)) {
                $to_keep[] = $tag;
                continue;
            }

            // Extract href
            if (preg_match('/href=["\']([^"\']+)["\']/i', $tag, $href_match)) {
                $to_process[] = [
                    'tag' => $tag,
                    'href' => $href_match[1],
                ];
            }
        }

        // Combine CSS if enabled
        if ($this->nitro->get_setting('css_combine') && count($to_process) > 1) {
            $combined = $this->combine_css($to_process);
            if ($combined) {
                // Remove all original tags
                foreach ($to_process as $item) {
                    $html = str_replace($item['tag'], '', $html);
                }
                // Add combined CSS
                $combined_tag = $this->create_css_tag($combined['url'], $combined['hash']);
                $html = str_replace('</head>', $combined_tag . "\n</head>", $html);
                return $html;
            }
        }

        // Process individual CSS files
        foreach ($to_process as $item) {
            $new_tag = $this->process_css_tag($item['tag'], $item['href']);
            $html = str_replace($item['tag'], $new_tag, $html);
        }

        return $html;
    }

    /**
     * Process individual CSS tag
     */
    private function process_css_tag(string $tag, string $href): string {
        $url = $href;

        // Minify if enabled
        if ($this->nitro->get_setting('css_minify')) {
            $minified = $this->minify_file($href);
            if ($minified) {
                $url = $minified;
            }
        }

        // Make async if enabled
        if ($this->nitro->get_setting('css_async')) {
            return $this->make_async($url);
        }

        // Just update URL if minified
        if ($url !== $href) {
            return str_replace($href, $url, $tag);
        }

        return $tag;
    }

    /**
     * Combine multiple CSS files
     */
    private function combine_css(array $items): ?array {
        $combined_content = '';
        $hash_parts = [];

        foreach ($items as $item) {
            $content = $this->get_css_content($item['href']);
            if ($content) {
                // Fix relative URLs in CSS
                $content = $this->fix_relative_urls($content, $item['href']);
                $combined_content .= "\n/* Source: {$item['href']} */\n" . $content;
                $hash_parts[] = md5($item['href']);
            }
        }

        if (empty($combined_content)) {
            return null;
        }

        // Minify combined content
        if ($this->nitro->get_setting('css_minify')) {
            $combined_content = $this->minify_css($combined_content);
        }

        // Generate filename
        $hash = md5(implode('', $hash_parts));
        $filename = 'combined-' . $hash . '.css';
        $filepath = $this->cache_dir . $filename;

        // Save combined file
        file_put_contents($filepath, $combined_content);

        return [
            'url' => $this->cache_url . $filename,
            'hash' => $hash,
        ];
    }

    /**
     * Minify a CSS file
     */
    private function minify_file(string $url): ?string {
        $content = $this->get_css_content($url);
        if (!$content) {
            return null;
        }

        // Fix relative URLs
        $content = $this->fix_relative_urls($content, $url);

        // Minify
        $minified = $this->minify_css($content);

        // Generate filename
        $hash = md5($url . $minified);
        $filename = 'min-' . $hash . '.css';
        $filepath = $this->cache_dir . $filename;

        // Check if already cached
        if (!file_exists($filepath)) {
            file_put_contents($filepath, $minified);
        }

        return $this->cache_url . $filename;
    }

    /**
     * Minify CSS content
     */
    private function minify_css(string $css): string {
        // Remove comments
        $css = preg_replace('/\/\*[\s\S]*?\*\//', '', $css);

        // Remove whitespace
        $css = preg_replace('/\s+/', ' ', $css);

        // Remove spaces around special characters
        $css = preg_replace('/\s*([\{\}\:\;\,\>])\s*/', '$1', $css);

        // Remove trailing semicolons before closing braces
        $css = str_replace(';}', '}', $css);

        // Remove newlines
        $css = str_replace(["\r\n", "\r", "\n"], '', $css);

        return trim($css);
    }

    /**
     * Get CSS content from URL
     */
    private function get_css_content(string $url): ?string {
        // Convert URL to path for local files
        $path = $this->url_to_path($url);

        if ($path && file_exists($path)) {
            return file_get_contents($path);
        }

        // Try to fetch remote CSS
        $response = wp_remote_get($url, [
            'timeout' => 10,
            'sslverify' => false,
        ]);

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            return wp_remote_retrieve_body($response);
        }

        return null;
    }

    /**
     * Convert URL to local file path
     */
    private function url_to_path(string $url): ?string {
        $site_url = site_url();
        $home_url = home_url();

        // Try site_url
        if (strpos($url, $site_url) === 0) {
            $path = ABSPATH . str_replace($site_url . '/', '', $url);
            if (file_exists($path)) {
                return $path;
            }
        }

        // Try home_url
        if (strpos($url, $home_url) === 0) {
            $path = ABSPATH . str_replace($home_url . '/', '', $url);
            if (file_exists($path)) {
                return $path;
            }
        }

        // Try relative URL
        if (strpos($url, '//') !== 0 && strpos($url, 'http') !== 0) {
            $path = ABSPATH . ltrim($url, '/');
            if (file_exists($path)) {
                return $path;
            }
        }

        return null;
    }

    /**
     * Fix relative URLs in CSS
     */
    private function fix_relative_urls(string $css, string $css_url): string {
        $base_url = dirname($css_url);

        // Fix url() references
        $css = preg_replace_callback(
            '/url\s*\(\s*["\']?\s*(?!data:|https?:|\/\/)(.*?)\s*["\']?\s*\)/i',
            function ($matches) use ($base_url) {
                $url = $matches[1];
                // Handle ../ paths
                $full_url = $this->resolve_relative_url($base_url, $url);
                return "url('{$full_url}')";
            },
            $css
        );

        return $css;
    }

    /**
     * Resolve relative URL
     */
    private function resolve_relative_url(string $base, string $relative): string {
        // Already absolute
        if (preg_match('/^(https?:)?\/\//i', $relative)) {
            return $relative;
        }

        // Parse base URL
        $base_parts = parse_url($base);
        $path = $base_parts['path'] ?? '/';

        if (strpos($relative, '/') === 0) {
            // Absolute path
            $path = $relative;
        } else {
            // Relative path
            $path = dirname($path) . '/' . $relative;
        }

        // Normalize path (resolve ../ and ./)
        $path = $this->normalize_path($path);

        $scheme = $base_parts['scheme'] ?? 'https';
        $host = $base_parts['host'] ?? parse_url(home_url(), PHP_URL_HOST);

        return "{$scheme}://{$host}{$path}";
    }

    /**
     * Normalize path (resolve ../ and ./)
     */
    private function normalize_path(string $path): string {
        $parts = explode('/', $path);
        $normalized = [];

        foreach ($parts as $part) {
            if ($part === '..') {
                array_pop($normalized);
            } elseif ($part !== '.' && $part !== '') {
                $normalized[] = $part;
            }
        }

        return '/' . implode('/', $normalized);
    }

    /**
     * Make CSS load asynchronously
     */
    private function make_async(string $url): string {
        $method = $this->nitro->get_setting('css_async_method', 'preload');

        if ($method === 'media_print') {
            // Media print trick
            return sprintf(
                '<link rel="stylesheet" href="%s" media="print" onload="this.media=\'all\'; this.onload=null;">',
                esc_url($url)
            );
        }

        // Preload method (default)
        return sprintf(
            '<link rel="preload" href="%s" as="style" onload="this.onload=null; this.rel=\'stylesheet\';">' .
            '<noscript><link rel="stylesheet" href="%s"></noscript>',
            esc_url($url),
            esc_url($url)
        );
    }

    /**
     * Create CSS tag
     */
    private function create_css_tag(string $url, string $hash): string {
        if ($this->nitro->get_setting('css_async')) {
            return $this->make_async($url);
        }

        return sprintf('<link rel="stylesheet" href="%s">', esc_url($url));
    }

    /**
     * Inject critical CSS
     */
    public function inject_critical_css(string $html): string {
        $critical_css = $this->nitro->get_setting('critical_css', '');
        
        if (empty($critical_css)) {
            return $html;
        }

        // Minify critical CSS
        $critical_css = $this->minify_css($critical_css);

        $style_tag = "<style id=\"enbombas-critical-css\">{$critical_css}</style>";

        // Inject after <head> or before first <link>
        if (preg_match('/<head[^>]*>/i', $html, $match)) {
            $html = str_replace($match[0], $match[0] . "\n" . $style_tag, $html);
        }

        return $html;
    }

    /**
     * Check if tag should be excluded
     */
    private function should_exclude(string $tag, array $excluded): bool {
        foreach ($excluded as $pattern) {
            if (empty($pattern)) continue;
            
            // Check if pattern exists in tag
            if (strpos($tag, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Remove unused CSS (basic implementation)
     */
    public function remove_unused(string $css, string $html): string {
        // Extract all classes and IDs from HTML
        $used_selectors = [];

        // Classes
        preg_match_all('/class=["\']([^"\']+)["\']/i', $html, $classes);
        if (!empty($classes[1])) {
            foreach ($classes[1] as $class_list) {
                $class_names = preg_split('/\s+/', $class_list);
                foreach ($class_names as $class) {
                    $used_selectors['.' . $class] = true;
                }
            }
        }

        // IDs
        preg_match_all('/id=["\']([^"\']+)["\']/i', $html, $ids);
        if (!empty($ids[1])) {
            foreach ($ids[1] as $id) {
                $used_selectors['#' . $id] = true;
            }
        }

        // Tags
        preg_match_all('/<([a-z][a-z0-9]*)[^>]*>/i', $html, $tags);
        if (!empty($tags[1])) {
            foreach ($tags[1] as $tag) {
                $used_selectors[strtolower($tag)] = true;
            }
        }

        // Parse CSS and remove unused rules (simplified)
        // This is a basic implementation - production would need a proper CSS parser
        $css = preg_replace_callback(
            '/([^{}]+)\{[^}]*\}/s',
            function ($matches) use ($used_selectors) {
                $selectors = $matches[1];
                
                // Keep @media, @keyframes, etc.
                if (strpos($selectors, '@') === 0) {
                    return $matches[0];
                }

                // Check if any selector is used
                $selector_list = explode(',', $selectors);
                foreach ($selector_list as $selector) {
                    $selector = trim($selector);
                    
                    // Extract main selector (first class, id, or tag)
                    if (preg_match('/^([.#]?[a-z][a-z0-9_-]*)/i', $selector, $main)) {
                        if (isset($used_selectors[$main[1]])) {
                            return $matches[0];
                        }
                    }
                }

                // Remove unused rule
                return '';
            },
            $css
        );

        return $css;
    }

    /**
     * Clear CSS cache
     */
    public function clear_cache(): int {
        $count = 0;

        if (is_dir($this->cache_dir)) {
            $files = glob($this->cache_dir . '*.css');
            foreach ($files as $file) {
                if (unlink($file)) {
                    $count++;
                }
            }
        }

        return $count;
    }
}



