<?php
/**
 * Next-Gen Image Formats Handler
 *
 * @package EnBombas\NextGen
 */

namespace EnBombas\NextGen;

use EnBombas\Compression\Compressor;
use EnBombas\Core\Plugin;

/**
 * Class NextGenFormats
 *
 * Handles WebP and AVIF generation and delivery.
 */
class NextGenFormats {
    /**
     * Compressor instance
     */
    private Compressor $compressor;

    /**
     * Constructor
     */
    public function __construct() {
        $this->compressor = new Compressor();
    }

    /**
     * Generate WebP version of an attachment
     */
    public function generate_webp(int $attachment_id): array {
        $plugin = Plugin::get_instance();
        $level = $plugin->get_setting('compression_level', 'normal');
        if ($level === 'smart') {
            $level = 'normal'; // Use normal for next-gen generation
        }

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return ['success' => false, 'message' => __('Original file not found.', 'en-bombas')];
        }

        // Generate WebP for main image
        $webp_path = $this->get_nextgen_path($file_path, 'webp');
        $result = $this->compressor->create_webp($file_path, $webp_path, $level);

        if (!$result['success']) {
            return $result;
        }

        $webp_files = [$webp_path];

        // Generate WebP for thumbnails
        $metadata = wp_get_attachment_metadata($attachment_id);
        if ($metadata && !empty($metadata['sizes'])) {
            $base_dir = dirname($file_path);

            foreach ($metadata['sizes'] as $size => $data) {
                $thumb_path = $base_dir . '/' . $data['file'];
                if (file_exists($thumb_path)) {
                    $thumb_webp = $this->get_nextgen_path($thumb_path, 'webp');
                    $thumb_result = $this->compressor->create_webp($thumb_path, $thumb_webp, $level);
                    if ($thumb_result['success']) {
                        $webp_files[] = $thumb_webp;
                    }
                }
            }
        }

        // Save WebP paths to meta
        update_post_meta($attachment_id, '_enbombas_webp', $webp_files);

        return [
            'success' => true,
            'message' => sprintf(__('Generated %d WebP files.', 'en-bombas'), count($webp_files)),
            'files' => $webp_files,
        ];
    }

    /**
     * Generate AVIF version of an attachment
     */
    public function generate_avif(int $attachment_id): array {
        if (!function_exists('imageavif')) {
            return [
                'success' => false,
                'message' => __('AVIF support requires PHP 8.1+ with GD.', 'en-bombas'),
            ];
        }

        $plugin = Plugin::get_instance();
        $level = $plugin->get_setting('compression_level', 'normal');
        if ($level === 'smart') {
            $level = 'normal';
        }

        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return ['success' => false, 'message' => __('Original file not found.', 'en-bombas')];
        }

        // Generate AVIF for main image
        $avif_path = $this->get_nextgen_path($file_path, 'avif');
        $result = $this->compressor->create_avif($file_path, $avif_path, $level);

        if (!$result['success']) {
            return $result;
        }

        $avif_files = [$avif_path];

        // Generate AVIF for thumbnails
        $metadata = wp_get_attachment_metadata($attachment_id);
        if ($metadata && !empty($metadata['sizes'])) {
            $base_dir = dirname($file_path);

            foreach ($metadata['sizes'] as $size => $data) {
                $thumb_path = $base_dir . '/' . $data['file'];
                if (file_exists($thumb_path)) {
                    $thumb_avif = $this->get_nextgen_path($thumb_path, 'avif');
                    $thumb_result = $this->compressor->create_avif($thumb_path, $thumb_avif, $level);
                    if ($thumb_result['success']) {
                        $avif_files[] = $thumb_avif;
                    }
                }
            }
        }

        // Save AVIF paths to meta
        update_post_meta($attachment_id, '_enbombas_avif', $avif_files);

        return [
            'success' => true,
            'message' => sprintf(__('Generated %d AVIF files.', 'en-bombas'), count($avif_files)),
            'files' => $avif_files,
        ];
    }

    /**
     * Get next-gen format file path
     */
    private function get_nextgen_path(string $original_path, string $format): string {
        $path_info = pathinfo($original_path);
        return $path_info['dirname'] . '/' . $path_info['filename'] . '.' . $format;
    }

    /**
     * Replace images with <picture> tags for next-gen format support
     */
    public function replace_with_picture_tags(string $content): string {
        if (empty($content)) {
            return $content;
        }

        // Match all img tags
        $pattern = '/<img([^>]+)>/i';

        return preg_replace_callback($pattern, function ($matches) {
            return $this->convert_img_to_picture($matches[0]);
        }, $content);
    }

    /**
     * Convert single img tag to picture tag
     */
    private function convert_img_to_picture(string $img_tag): string {
        // Extract src attribute
        if (!preg_match('/src=["\']([^"\']+)["\']/i', $img_tag, $src_match)) {
            return $img_tag;
        }

        $original_src = $src_match[1];
        $upload_dir = wp_upload_dir();

        // Check if it's from our uploads
        if (strpos($original_src, $upload_dir['baseurl']) === false) {
            return $img_tag;
        }

        // Convert URL to path
        $original_path = str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $original_src);

        // Check for next-gen versions
        $webp_path = $this->get_nextgen_path($original_path, 'webp');
        $avif_path = $this->get_nextgen_path($original_path, 'avif');

        $has_webp = file_exists($webp_path);
        $has_avif = file_exists($avif_path);

        // If no next-gen versions exist, return original
        if (!$has_webp && !$has_avif) {
            return $img_tag;
        }

        // Build picture tag
        $sources = [];

        if ($has_avif) {
            $avif_url = str_replace($upload_dir['basedir'], $upload_dir['baseurl'], $avif_path);
            $sources[] = sprintf('<source srcset="%s" type="image/avif">', esc_url($avif_url));
        }

        if ($has_webp) {
            $webp_url = str_replace($upload_dir['basedir'], $upload_dir['baseurl'], $webp_path);
            $sources[] = sprintf('<source srcset="%s" type="image/webp">', esc_url($webp_url));
        }

        // Build the picture element
        $picture = '<picture>' . "\n";
        $picture .= implode("\n", $sources) . "\n";
        $picture .= $img_tag . "\n";
        $picture .= '</picture>';

        return $picture;
    }

    /**
     * Update .htaccess for WebP/AVIF rewrite rules
     */
    public function update_htaccess(): bool {
        $htaccess_file = ABSPATH . '.htaccess';

        if (!is_writable($htaccess_file)) {
            return false;
        }

        $rules = $this->get_htaccess_rules();
        $current_content = file_get_contents($htaccess_file);

        // Remove existing rules
        $current_content = $this->remove_htaccess_rules($current_content);

        // Add new rules
        $new_content = $rules . "\n\n" . $current_content;

        return (bool) file_put_contents($htaccess_file, $new_content);
    }

    /**
     * Get .htaccess rewrite rules for next-gen formats
     */
    private function get_htaccess_rules(): string {
        $plugin = Plugin::get_instance();
        $generate_webp = $plugin->get_setting('generate_webp');
        $generate_avif = $plugin->get_setting('generate_avif');

        $rules = "# BEGIN En Bombas Next-Gen Images\n";
        $rules .= "<IfModule mod_rewrite.c>\n";
        $rules .= "RewriteEngine On\n\n";

        if ($generate_avif) {
            $rules .= "# Serve AVIF if browser supports it and file exists\n";
            $rules .= "RewriteCond %{HTTP_ACCEPT} image/avif\n";
            $rules .= "RewriteCond %{REQUEST_FILENAME} \".(jpe?g|png|gif)$\" [NC]\n";
            $rules .= "RewriteCond %{REQUEST_FILENAME}.avif -f\n";
            $rules .= "RewriteRule ^(.+)\\.(jpe?g|png|gif)$ $1.$2.avif [T=image/avif,E=REQUEST_image]\n\n";
        }

        if ($generate_webp) {
            $rules .= "# Serve WebP if browser supports it and file exists\n";
            $rules .= "RewriteCond %{HTTP_ACCEPT} image/webp\n";
            $rules .= "RewriteCond %{REQUEST_FILENAME} \".(jpe?g|png|gif)$\" [NC]\n";
            $rules .= "RewriteCond %{REQUEST_FILENAME}.webp -f\n";
            $rules .= "RewriteRule ^(.+)\\.(jpe?g|png|gif)$ $1.$2.webp [T=image/webp,E=REQUEST_image]\n\n";
        }

        $rules .= "</IfModule>\n";

        // Add headers
        $rules .= "<IfModule mod_headers.c>\n";
        $rules .= "Header append Vary Accept env=REQUEST_image\n";
        $rules .= "</IfModule>\n";

        // Add MIME types
        $rules .= "<IfModule mod_mime.c>\n";
        if ($generate_webp) {
            $rules .= "AddType image/webp .webp\n";
        }
        if ($generate_avif) {
            $rules .= "AddType image/avif .avif\n";
        }
        $rules .= "</IfModule>\n";
        $rules .= "# END En Bombas Next-Gen Images\n";

        return $rules;
    }

    /**
     * Remove En Bombas rules from .htaccess content
     */
    private function remove_htaccess_rules(string $content): string {
        $pattern = '/# BEGIN En Bombas Next-Gen Images.*?# END En Bombas Next-Gen Images\s*/s';
        return preg_replace($pattern, '', $content);
    }

    /**
     * Delete next-gen versions for an attachment
     */
    public function delete_nextgen_files(int $attachment_id): void {
        // Delete WebP files
        $webp_files = get_post_meta($attachment_id, '_enbombas_webp', true);
        if (is_array($webp_files)) {
            foreach ($webp_files as $file) {
                if (file_exists($file)) {
                    @unlink($file);
                }
            }
        }

        // Delete AVIF files
        $avif_files = get_post_meta($attachment_id, '_enbombas_avif', true);
        if (is_array($avif_files)) {
            foreach ($avif_files as $file) {
                if (file_exists($file)) {
                    @unlink($file);
                }
            }
        }

        // Clean up meta
        delete_post_meta($attachment_id, '_enbombas_webp');
        delete_post_meta($attachment_id, '_enbombas_avif');
    }

    /**
     * Check if attachment has next-gen versions
     */
    public function has_nextgen_versions(int $attachment_id): array {
        return [
            'webp' => !empty(get_post_meta($attachment_id, '_enbombas_webp', true)),
            'avif' => !empty(get_post_meta($attachment_id, '_enbombas_avif', true)),
        ];
    }
}



