<?php
/**
 * AI Monitoring Class
 *
 * @package EnBombas\Monitoring
 */

namespace EnBombas\Monitoring;

use EnBombas\Core\Plugin;

/**
 * Class AIMonitor
 *
 * AI-powered image monitoring and alerts.
 */
class AIMonitor {
    /**
     * Option key for monitoring data
     */
    private const MONITORING_OPTION = 'enbombas_monitoring_data';

    /**
     * Alert types
     */
    public const ALERT_UNOPTIMIZED = 'unoptimized';
    public const ALERT_OVERSIZED = 'oversized';
    public const ALERT_QUEUE_WARNING = 'queue_warning';
    public const ALERT_LARGE_BANNER = 'large_banner';

    /**
     * Run monitoring scan
     */
    public function run_scan(): array {
        $plugin = Plugin::get_instance();
        $settings = $plugin->get_settings();

        if (!$settings['monitoring_enabled']) {
            return ['success' => false, 'message' => __('El monitoreo está deshabilitado.', 'en-bombas')];
        }

        $alerts = [];
        $stats = [];

        // Check for unoptimized images
        if ($settings['alert_unoptimized']) {
            $unoptimized = $this->scan_unoptimized();
            if ($unoptimized['count'] > 0) {
                $alerts[] = [
                    'type' => self::ALERT_UNOPTIMIZED,
                    'severity' => $this->get_severity($unoptimized['count'], [10, 50, 100]),
                    'count' => $unoptimized['count'],
                    'message' => sprintf(
                        __('Se detectaron %d imágenes sin optimizar. Considera ejecutar la optimización masiva, parce.', 'en-bombas'),
                        $unoptimized['count']
                    ),
                    'action' => 'bulk_optimize',
                ];
            }
            $stats['unoptimized'] = $unoptimized;
        }

        // Check for oversized images
        if ($settings['alert_oversized']) {
            $oversized = $this->scan_oversized($settings['oversized_threshold']);
            if ($oversized['count'] > 0) {
                $alerts[] = [
                    'type' => self::ALERT_OVERSIZED,
                    'severity' => $this->get_severity($oversized['count'], [5, 20, 50]),
                    'count' => $oversized['count'],
                    'message' => sprintf(
                        __('%d imágenes exceden %dKB. Las imágenes muy pesadas hacen lento tu sitio.', 'en-bombas'),
                        $oversized['count'],
                        $settings['oversized_threshold']
                    ),
                    'action' => 'view_oversized',
                    'items' => $oversized['items'],
                ];
            }
            $stats['oversized'] = $oversized;
        }

        // Check for large banners/hero images
        $large_banners = $this->scan_large_banners();
        if ($large_banners['count'] > 0) {
            $alerts[] = [
                'type' => self::ALERT_LARGE_BANNER,
                'severity' => 'warning',
                'count' => $large_banners['count'],
                'message' => sprintf(
                    __('Se detectaron %d banners que exceden 1920px de ancho. Considera redimensionarlos para mejor rendimiento.', 'en-bombas'),
                    $large_banners['count']
                ),
                'action' => 'view_banners',
                'items' => $large_banners['items'],
            ];
        }
        $stats['large_banners'] = $large_banners;

        // Queue warning
        $queue_status = $this->check_queue_status();
        if ($queue_status['warning']) {
            $alerts[] = [
                'type' => self::ALERT_QUEUE_WARNING,
                'severity' => 'warning',
                'message' => $queue_status['message'],
                'action' => 'manage_queue',
            ];
        }

        // Save monitoring data
        $monitoring_data = [
            'last_scan' => current_time('mysql'),
            'alerts' => $alerts,
            'stats' => $stats,
            'health_score' => $this->calculate_health_score($stats),
        ];
        update_option(self::MONITORING_OPTION, $monitoring_data);

        return [
            'success' => true,
            'data' => $monitoring_data,
        ];
    }

    /**
     * Scan for unoptimized images
     */
    private function scan_unoptimized(): array {
        global $wpdb;

        $count = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_enbombas_optimized'
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE 'image/%'
            AND (pm.meta_value IS NULL OR pm.meta_value != '1')
        ");

        // Get sample of unoptimized images
        $sample = $wpdb->get_col("
            SELECT p.ID 
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_enbombas_optimized'
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE 'image/%'
            AND (pm.meta_value IS NULL OR pm.meta_value != '1')
            LIMIT 10
        ");

        return [
            'count' => (int) $count,
            'sample' => $sample,
        ];
    }

    /**
     * Scan for oversized images
     */
    private function scan_oversized(int $threshold_kb): array {
        global $wpdb;

        $threshold_bytes = $threshold_kb * 1024;
        $upload_dir = wp_upload_dir();
        $oversized = [];

        // Get recent images
        $attachments = $wpdb->get_results("
            SELECT ID, post_title, guid
            FROM {$wpdb->posts}
            WHERE post_type = 'attachment'
            AND post_mime_type LIKE 'image/%'
            ORDER BY post_date DESC
            LIMIT 500
        ");

        foreach ($attachments as $attachment) {
            $file_path = get_attached_file($attachment->ID);
            if ($file_path && file_exists($file_path)) {
                $size = filesize($file_path);
                if ($size > $threshold_bytes) {
                    $oversized[] = [
                        'id' => $attachment->ID,
                        'title' => $attachment->post_title,
                        'size' => $size,
                        'size_formatted' => size_format($size),
                        'url' => $attachment->guid,
                    ];
                }
            }
        }

        return [
            'count' => count($oversized),
            'items' => array_slice($oversized, 0, 20), // Return top 20
        ];
    }

    /**
     * Scan for large banner images
     */
    private function scan_large_banners(): array {
        global $wpdb;

        $large_banners = [];
        $max_width = 1920;

        // Get recent larger images
        $attachments = $wpdb->get_results("
            SELECT p.ID, p.post_title, pm.meta_value as metadata
            FROM {$wpdb->posts} p
            JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'attachment'
            AND p.post_mime_type LIKE 'image/%'
            AND pm.meta_key = '_wp_attachment_metadata'
            ORDER BY p.post_date DESC
            LIMIT 200
        ");

        foreach ($attachments as $attachment) {
            $metadata = maybe_unserialize($attachment->metadata);
            if (!$metadata || empty($metadata['width'])) {
                continue;
            }

            // Check if image is wider than threshold
            if ($metadata['width'] > $max_width) {
                $large_banners[] = [
                    'id' => $attachment->ID,
                    'title' => $attachment->post_title,
                    'width' => $metadata['width'],
                    'height' => $metadata['height'] ?? 0,
                    'dimensions' => sprintf('%dx%d', $metadata['width'], $metadata['height'] ?? 0),
                ];
            }
        }

        return [
            'count' => count($large_banners),
            'items' => array_slice($large_banners, 0, 10),
        ];
    }

    /**
     * Check queue status
     */
    private function check_queue_status(): array {
        $queue = get_option('enbombas_bulk_queue', []);
        $queue_count = count($queue);

        if ($queue_count > 100) {
            return [
                'warning' => true,
                'message' => sprintf(
                    __('Se detectó una cola de optimización grande (%d imágenes). Esto puede tardar un rato en procesarse.', 'en-bombas'),
                    $queue_count
                ),
                'count' => $queue_count,
            ];
        }

        return [
            'warning' => false,
            'count' => $queue_count,
        ];
    }

    /**
     * Get severity level based on count
     */
    private function get_severity(int $count, array $thresholds): string {
        if ($count >= $thresholds[2]) {
            return 'critical';
        }
        if ($count >= $thresholds[1]) {
            return 'high';
        }
        if ($count >= $thresholds[0]) {
            return 'medium';
        }
        return 'low';
    }

    /**
     * Calculate overall health score (0-100)
     */
    private function calculate_health_score(array $stats): int {
        $plugin = Plugin::get_instance();
        $total_stats = $plugin->get_statistics();

        $score = 100;

        // Deduct points for unoptimized images
        $unoptimized_ratio = $total_stats['total_images'] > 0
            ? ($stats['unoptimized']['count'] ?? 0) / $total_stats['total_images']
            : 0;
        $score -= $unoptimized_ratio * 50;

        // Deduct points for oversized images
        $oversized_count = $stats['oversized']['count'] ?? 0;
        if ($oversized_count > 50) {
            $score -= 20;
        } elseif ($oversized_count > 20) {
            $score -= 10;
        } elseif ($oversized_count > 5) {
            $score -= 5;
        }

        // Deduct points for large banners
        $banner_count = $stats['large_banners']['count'] ?? 0;
        if ($banner_count > 10) {
            $score -= 10;
        } elseif ($banner_count > 5) {
            $score -= 5;
        }

        return max(0, min(100, (int) round($score)));
    }

    /**
     * Get monitoring data
     */
    public function get_monitoring_data(): array {
        $data = get_option(self::MONITORING_OPTION, []);

        if (empty($data)) {
            // Run initial scan
            $result = $this->run_scan();
            return $result['data'] ?? [];
        }

        return $data;
    }

    /**
     * Get alerts
     */
    public function get_alerts(): array {
        $data = $this->get_monitoring_data();
        return $data['alerts'] ?? [];
    }

    /**
     * Get health score
     */
    public function get_health_score(): int {
        $data = $this->get_monitoring_data();
        return $data['health_score'] ?? 100;
    }

    /**
     * Get last scan time
     */
    public function get_last_scan(): ?string {
        $data = get_option(self::MONITORING_OPTION, []);
        return $data['last_scan'] ?? null;
    }

    /**
     * Clear monitoring data
     */
    public function clear_data(): void {
        delete_option(self::MONITORING_OPTION);
    }

    /**
     * Get AI status message
     */
    public function get_ai_status(): array {
        $health_score = $this->get_health_score();
        $alerts = $this->get_alerts();

        if ($health_score >= 90) {
            return [
                'status' => 'excellent',
                'icon' => 'rocket',
                'title' => __('Motor En Bombas: MODO TURBO', 'en-bombas'),
                'message' => __('¡Tus imágenes están completamente optimizadas! El rendimiento del sitio está al máximo.', 'en-bombas'),
                'color' => '#10b981',
            ];
        }

        if ($health_score >= 70) {
            return [
                'status' => 'good',
                'icon' => 'bolt',
                'title' => __('Motor En Bombas: Activo', 'en-bombas'),
                'message' => __('Buen estado de optimización. Algunas imágenes podrían necesitar atención.', 'en-bombas'),
                'color' => '#7A57D1',
            ];
        }

        if ($health_score >= 50) {
            return [
                'status' => 'warning',
                'icon' => 'alert',
                'title' => __('Motor En Bombas: Necesita Combustible', 'en-bombas'),
                'message' => __('Se detectaron varias imágenes sin optimizar. Ejecuta la optimización masiva para mejorar el rendimiento.', 'en-bombas'),
                'color' => '#f59e0b',
            ];
        }

        return [
            'status' => 'critical',
            'icon' => 'x',
            'title' => __('Motor En Bombas: Detenido', 'en-bombas'),
            'message' => __('Muchas imágenes siguen pesadas, parce. Esto puede bajar la velocidad de tu sitio.', 'en-bombas'),
            'color' => '#ef4444',
        ];
    }
}
