<?php
/**
 * Main Plugin Class
 *
 * @package EnBombas\Core
 */

namespace EnBombas\Core;

use EnBombas\AdminUI\AdminPages;
use EnBombas\AdminUI\AdminBar;
use EnBombas\AdminUI\MediaLibrary;
use EnBombas\AdminUI\RestAPI;
use EnBombas\Optimizer\ImageOptimizer;
use EnBombas\Bulk\BulkOptimizer;
use EnBombas\Monitoring\AIMonitor;
use EnBombas\Backup\BackupManager;
use EnBombas\NextGen\NextGenFormats;
use EnBombas\Nitro\NitroCore;

/**
 * Class Plugin
 *
 * Core plugin class implementing singleton pattern.
 */
class Plugin {
    /**
     * Singleton instance
     */
    private static ?Plugin $instance = null;

    /**
     * Plugin settings
     */
    private array $settings = [];

    /**
     * Admin pages handler
     */
    private ?AdminPages $admin_pages = null;

    /**
     * Admin bar handler
     */
    private ?AdminBar $admin_bar = null;

    /**
     * Media library handler
     */
    private ?MediaLibrary $media_library = null;

    /**
     * Image optimizer
     */
    private ?ImageOptimizer $optimizer = null;

    /**
     * Bulk optimizer
     */
    private ?BulkOptimizer $bulk_optimizer = null;

    /**
     * AI Monitor
     */
    private ?AIMonitor $ai_monitor = null;

    /**
     * Backup manager
     */
    private ?BackupManager $backup_manager = null;

    /**
     * Next-gen formats handler
     */
    private ?NextGenFormats $nextgen = null;

    /**
     * Nitro performance module
     */
    private ?NitroCore $nitro = null;

    /**
     * Private constructor for singleton
     */
    private function __construct() {
        $this->load_settings();
    }

    /**
     * Get singleton instance
     */
    public static function get_instance(): Plugin {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Load plugin settings
     */
    private function load_settings(): void {
        $defaults = self::get_default_settings();
        $saved = get_option('enbombas_settings', []);
        $this->settings = wp_parse_args($saved, $defaults);
    }

    /**
     * Get default settings
     */
    public static function get_default_settings(): array {
        return [
            // Compression settings
            'compression_level' => 'smart', // normal, high, ultra, smart
            'auto_optimize' => true,
            'remove_exif' => true,
            'max_width' => 2560,
            'max_height' => 2560,
            'resize_larger' => true,
            'optimize_thumbnails' => true,

            // Next-gen formats
            'generate_webp' => true,
            'generate_avif' => false,
            'nextgen_method' => 'picture', // picture, htaccess

            // Backup
            'backup_enabled' => true,
            'backup_location' => 'default', // default, custom

            // Custom folders
            'custom_folders' => [],

            // AI Monitoring
            'monitoring_enabled' => true,
            'alert_oversized' => true,
            'oversized_threshold' => 500, // KB
            'alert_unoptimized' => true,

            // Bulk settings
            'turbo_mode' => false,
            'parallel_processes' => 2,

            // Statistics
            'total_optimized' => 0,
            'total_saved_bytes' => 0,
        ];
    }

    /**
     * Get setting value
     */
    public function get_setting(string $key, mixed $default = null): mixed {
        return $this->settings[$key] ?? $default;
    }

    /**
     * Update setting
     */
    public function update_setting(string $key, mixed $value): void {
        $this->settings[$key] = $value;
        update_option('enbombas_settings', $this->settings);
    }

    /**
     * Update multiple settings
     */
    public function update_settings(array $settings): void {
        $this->settings = wp_parse_args($settings, $this->settings);
        update_option('enbombas_settings', $this->settings);
    }

    /**
     * Get all settings
     */
    public function get_settings(): array {
        return $this->settings;
    }

    /**
     * Run the plugin
     */
    public function run(): void {
        // Initialize components
        $this->init_components();

        // Register hooks
        $this->register_hooks();
    }

    /**
     * Initialize plugin components
     */
    private function init_components(): void {
        $this->backup_manager = new BackupManager();
        $this->optimizer = new ImageOptimizer($this->backup_manager);
        $this->nextgen = new NextGenFormats();
        $this->bulk_optimizer = new BulkOptimizer($this->optimizer);
        $this->ai_monitor = new AIMonitor();

        // Initialize Nitro performance module
        $this->nitro = NitroCore::get_instance();
        $this->nitro->init();

        if (is_admin()) {
            $this->admin_pages = new AdminPages();
            $this->media_library = new MediaLibrary($this->optimizer, $this->backup_manager);
        }

        // Initialize admin bar (works on frontend and backend)
        $this->admin_bar = new AdminBar();

        // Initialize REST API
        new RestAPI();
    }

    /**
     * Register WordPress hooks
     */
    private function register_hooks(): void {
        // Auto-optimize on upload
        if ($this->get_setting('auto_optimize')) {
            add_filter('wp_generate_attachment_metadata', [$this, 'auto_optimize_upload'], 10, 2);
        }

        // Next-gen format delivery
        if ($this->get_setting('generate_webp') || $this->get_setting('generate_avif')) {
            if ($this->get_setting('nextgen_method') === 'picture') {
                add_filter('the_content', [$this->nextgen, 'replace_with_picture_tags']);
                add_filter('post_thumbnail_html', [$this->nextgen, 'replace_with_picture_tags']);
            }
        }

        // AJAX handlers
        add_action('wp_ajax_enbombas_optimize_single', [$this, 'ajax_optimize_single']);
        add_action('wp_ajax_enbombas_restore_single', [$this, 'ajax_restore_single']);
        add_action('wp_ajax_enbombas_bulk_optimize', [$this, 'ajax_bulk_optimize']);
        add_action('wp_ajax_enbombas_get_stats', [$this, 'ajax_get_stats']);
        add_action('wp_ajax_enbombas_save_settings', [$this, 'ajax_save_settings']);
        add_action('wp_ajax_enbombas_get_monitoring_data', [$this, 'ajax_get_monitoring_data']);
        add_action('wp_ajax_enbombas_get_bulk_status', [$this, 'ajax_get_bulk_status']);

        // Cron hooks
        add_action('enbombas_cron_monitoring', [$this->ai_monitor, 'run_scan']);

        // Schedule cron if not scheduled
        if (!wp_next_scheduled('enbombas_cron_monitoring') && $this->get_setting('monitoring_enabled')) {
            wp_schedule_event(time(), 'hourly', 'enbombas_cron_monitoring');
        }
    }

    /**
     * Auto-optimize uploaded images
     */
    public function auto_optimize_upload(array $metadata, int $attachment_id): array {
        if (!wp_attachment_is_image($attachment_id)) {
            return $metadata;
        }

        // Optimize main image
        $this->optimizer->optimize($attachment_id);

        // Generate next-gen formats
        if ($this->get_setting('generate_webp')) {
            $this->nextgen->generate_webp($attachment_id);
        }

        if ($this->get_setting('generate_avif')) {
            $this->nextgen->generate_avif($attachment_id);
        }

        return $metadata;
    }

    /**
     * AJAX: Optimize single image
     */
    public function ajax_optimize_single(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['message' => __('Permission denied.', 'en-bombas')]);
        }

        $attachment_id = absint($_POST['attachment_id'] ?? 0);
        if (!$attachment_id) {
            wp_send_json_error(['message' => __('Invalid attachment ID.', 'en-bombas')]);
        }

        $result = $this->optimizer->optimize($attachment_id);

        if ($result['success']) {
            // Generate next-gen formats
            if ($this->get_setting('generate_webp')) {
                $this->nextgen->generate_webp($attachment_id);
            }
            if ($this->get_setting('generate_avif')) {
                $this->nextgen->generate_avif($attachment_id);
            }

            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX: Restore single image
     */
    public function ajax_restore_single(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['message' => __('Permission denied.', 'en-bombas')]);
        }

        $attachment_id = absint($_POST['attachment_id'] ?? 0);
        if (!$attachment_id) {
            wp_send_json_error(['message' => __('Invalid attachment ID.', 'en-bombas')]);
        }

        $result = $this->backup_manager->restore($attachment_id);
        wp_send_json($result);
    }

    /**
     * AJAX: Bulk optimize
     */
    public function ajax_bulk_optimize(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['message' => __('Permission denied.', 'en-bombas')]);
        }

        $action = sanitize_text_field($_POST['bulk_action'] ?? 'start');

        switch ($action) {
            case 'start':
                $result = $this->bulk_optimizer->start();
                break;
            case 'process':
                $result = $this->bulk_optimizer->process_batch();
                break;
            case 'stop':
                $result = $this->bulk_optimizer->stop();
                break;
            default:
                $result = ['success' => false, 'message' => __('Invalid action.', 'en-bombas')];
        }

        wp_send_json($result);
    }

    /**
     * AJAX: Get bulk optimization status
     */
    public function ajax_get_bulk_status(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['message' => __('Permission denied.', 'en-bombas')]);
        }

        $status = $this->bulk_optimizer->get_status();
        wp_send_json_success($status);
    }

    /**
     * AJAX: Get statistics
     */
    public function ajax_get_stats(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['message' => __('Permission denied.', 'en-bombas')]);
        }

        $stats = $this->get_statistics();
        wp_send_json_success($stats);
    }

    /**
     * AJAX: Save settings
     */
    public function ajax_save_settings(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Permission denied.', 'en-bombas')]);
        }

        $settings = [];

        // Sanitize each setting
        $settings['compression_level'] = sanitize_text_field($_POST['compression_level'] ?? 'smart');
        $settings['auto_optimize'] = isset($_POST['auto_optimize']);
        $settings['remove_exif'] = isset($_POST['remove_exif']);
        $settings['max_width'] = absint($_POST['max_width'] ?? 2560);
        $settings['max_height'] = absint($_POST['max_height'] ?? 2560);
        $settings['resize_larger'] = isset($_POST['resize_larger']);
        $settings['optimize_thumbnails'] = isset($_POST['optimize_thumbnails']);
        $settings['generate_webp'] = isset($_POST['generate_webp']);
        $settings['generate_avif'] = isset($_POST['generate_avif']);
        $settings['nextgen_method'] = sanitize_text_field($_POST['nextgen_method'] ?? 'picture');
        $settings['backup_enabled'] = isset($_POST['backup_enabled']);
        $settings['monitoring_enabled'] = isset($_POST['monitoring_enabled']);
        $settings['alert_oversized'] = isset($_POST['alert_oversized']);
        $settings['oversized_threshold'] = absint($_POST['oversized_threshold'] ?? 500);
        $settings['turbo_mode'] = isset($_POST['turbo_mode']);
        $settings['parallel_processes'] = min(5, max(1, absint($_POST['parallel_processes'] ?? 2)));

        // Custom folders
        if (isset($_POST['custom_folders'])) {
            $folders = array_map('sanitize_text_field', (array) $_POST['custom_folders']);
            $settings['custom_folders'] = array_filter($folders);
        }

        $this->update_settings($settings);

        // Update htaccess if needed
        if ($settings['generate_webp'] && $settings['nextgen_method'] === 'htaccess') {
            $this->nextgen->update_htaccess();
        }

        wp_send_json_success(['message' => __('Settings saved successfully!', 'en-bombas')]);
    }

    /**
     * AJAX: Get monitoring data
     */
    public function ajax_get_monitoring_data(): void {
        check_ajax_referer('enbombas_nonce', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_send_json_error(['message' => __('Permission denied.', 'en-bombas')]);
        }

        $data = $this->ai_monitor->get_monitoring_data();
        wp_send_json_success($data);
    }

    /**
     * Get plugin statistics
     */
    public function get_statistics(): array {
        global $wpdb;

        // Get total images
        $total_images = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_mime_type LIKE 'image/%'"
        );

        // Get optimized images
        $optimized = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = %s",
                '_enbombas_optimized',
                '1'
            )
        );

        // Get total bytes saved
        $total_saved = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT SUM(meta_value) FROM {$wpdb->postmeta} WHERE meta_key = %s",
                '_enbombas_saved_bytes'
            )
        );

        // Get monthly stats
        $month_start = date('Y-m-01 00:00:00');
        $monthly_optimized = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->postmeta} pm
                JOIN {$wpdb->posts} p ON pm.post_id = p.ID
                WHERE pm.meta_key = %s AND pm.meta_value = %s
                AND p.post_modified >= %s",
                '_enbombas_optimized',
                '1',
                $month_start
            )
        );

        return [
            'total_images' => (int) $total_images,
            'optimized_images' => (int) $optimized,
            'unoptimized_images' => (int) $total_images - (int) $optimized,
            'total_saved_bytes' => (int) $total_saved,
            'total_saved_formatted' => size_format((int) $total_saved),
            'monthly_optimized' => (int) $monthly_optimized,
            'optimization_rate' => $total_images > 0 ? round(($optimized / $total_images) * 100, 1) : 0,
        ];
    }

    /**
     * Plugin activation
     */
    public static function activate(): void {
        // Create backup directory
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/enbombas-backups';

        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);

            // Add index.php for security
            file_put_contents($backup_dir . '/index.php', '<?php // Silence is golden');

            // Add .htaccess to deny direct access
            file_put_contents($backup_dir . '/.htaccess', 'deny from all');
        }

        // Set default options
        if (!get_option('enbombas_settings')) {
            add_option('enbombas_settings', self::get_default_settings());
        }

        // Store version
        update_option('enbombas_version', ENBOMBAS_VERSION);

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public static function deactivate(): void {
        // Clear scheduled cron events
        wp_clear_scheduled_hook('enbombas_cron_optimize');
        wp_clear_scheduled_hook('enbombas_cron_monitoring');

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Get optimizer instance
     */
    public function get_optimizer(): ImageOptimizer {
        return $this->optimizer;
    }

    /**
     * Get backup manager instance
     */
    public function get_backup_manager(): BackupManager {
        return $this->backup_manager;
    }

    /**
     * Get bulk optimizer instance
     */
    public function get_bulk_optimizer(): BulkOptimizer {
        return $this->bulk_optimizer;
    }

    /**
     * Get AI monitor instance
     */
    public function get_ai_monitor(): AIMonitor {
        return $this->ai_monitor;
    }

    /**
     * Get Nitro module instance
     */
    public function get_nitro(): NitroCore {
        return $this->nitro;
    }
}

