<?php
/**
 * Image Compressor Class
 *
 * @package EnBombas\Compression
 */

namespace EnBombas\Compression;

/**
 * Class Compressor
 *
 * Handles image compression with multiple quality levels.
 */
class Compressor {
    /**
     * Quality presets for different compression levels
     */
    private const QUALITY_PRESETS = [
        'normal' => [
            'jpg' => 82,
            'png' => 7, // PNG compression level 0-9
            'gif' => 85,
            'webp' => 82,
            'avif' => 65,
        ],
        'high' => [
            'jpg' => 72,
            'png' => 8,
            'gif' => 75,
            'webp' => 72,
            'avif' => 55,
        ],
        'ultra' => [
            'jpg' => 60,
            'png' => 9,
            'gif' => 65,
            'webp' => 60,
            'avif' => 45,
        ],
    ];

    /**
     * Current compression level
     */
    private string $level = 'normal';

    /**
     * Whether to remove EXIF data
     */
    private bool $remove_exif = true;

    /**
     * Set compression level
     */
    public function set_level(string $level): self {
        if (isset(self::QUALITY_PRESETS[$level]) || $level === 'smart') {
            $this->level = $level;
        }
        return $this;
    }

    /**
     * Set EXIF removal option
     */
    public function set_remove_exif(bool $remove): self {
        $this->remove_exif = $remove;
        return $this;
    }

    /**
     * Compress an image file
     *
     * @param string $file_path Path to the image file
     * @return array Result with success status and data
     */
    public function compress(string $file_path): array {
        if (!file_exists($file_path)) {
            return [
                'success' => false,
                'message' => __('File not found.', 'en-bombas'),
            ];
        }

        $original_size = filesize($file_path);
        $mime_type = wp_check_filetype($file_path)['type'];

        // Determine compression level (smart mode)
        $level = $this->level;
        if ($level === 'smart') {
            $level = $this->analyze_image_for_compression($file_path, $mime_type);
        }

        $result = match ($mime_type) {
            'image/jpeg', 'image/jpg' => $this->compress_jpeg($file_path, $level),
            'image/png' => $this->compress_png($file_path, $level),
            'image/gif' => $this->compress_gif($file_path, $level),
            'application/pdf' => $this->compress_pdf($file_path, $level),
            default => ['success' => false, 'message' => __('Unsupported file type.', 'en-bombas')],
        };

        if ($result['success']) {
            clearstatcache(true, $file_path);
            $new_size = filesize($file_path);
            $saved = $original_size - $new_size;
            $percentage = $original_size > 0 ? round(($saved / $original_size) * 100, 1) : 0;

            $result['original_size'] = $original_size;
            $result['new_size'] = $new_size;
            $result['saved_bytes'] = $saved;
            $result['saved_percentage'] = $percentage;
            $result['compression_level'] = $level;
        }

        return $result;
    }

    /**
     * Analyze image to determine optimal compression level (Smart Bomb Mode)
     */
    private function analyze_image_for_compression(string $file_path, string $mime_type): string {
        // Get image dimensions
        $image_data = @getimagesize($file_path);
        if (!$image_data) {
            return 'normal';
        }

        $width = $image_data[0];
        $height = $image_data[1];
        $total_pixels = $width * $height;

        // Analyze color complexity for JPEG/PNG
        if (in_array($mime_type, ['image/jpeg', 'image/jpg', 'image/png'])) {
            $complexity = $this->calculate_image_complexity($file_path, $mime_type);

            // High complexity (photos, detailed graphics) - use normal compression
            if ($complexity > 0.7) {
                return 'normal';
            }

            // Medium complexity - use high compression
            if ($complexity > 0.4) {
                return 'high';
            }

            // Low complexity (flat graphics, icons) - use ultra compression
            return 'ultra';
        }

        // For GIFs, consider file size
        $file_size = filesize($file_path);
        if ($file_size > 500000) { // > 500KB
            return 'high';
        }

        return 'normal';
    }

    /**
     * Calculate image complexity score (0-1)
     */
    private function calculate_image_complexity(string $file_path, string $mime_type): float {
        try {
            $image = match ($mime_type) {
                'image/jpeg', 'image/jpg' => @imagecreatefromjpeg($file_path),
                'image/png' => @imagecreatefrompng($file_path),
                default => null,
            };

            if (!$image) {
                return 0.5; // Default to medium complexity
            }

            $width = imagesx($image);
            $height = imagesy($image);

            // Sample pixels to analyze color variance
            $sample_size = min(1000, $width * $height);
            $colors = [];

            for ($i = 0; $i < $sample_size; $i++) {
                $x = rand(0, $width - 1);
                $y = rand(0, $height - 1);
                $rgb = imagecolorat($image, $x, $y);
                $colors[] = $rgb;
            }

            imagedestroy($image);

            // Calculate unique colors ratio
            $unique_colors = count(array_unique($colors));
            $color_ratio = $unique_colors / $sample_size;

            // Calculate variance
            $variance = $this->calculate_variance($colors);
            $normalized_variance = min(1, $variance / 1000000000);

            // Combined complexity score
            return ($color_ratio * 0.6) + ($normalized_variance * 0.4);
        } catch (\Exception $e) {
            return 0.5;
        }
    }

    /**
     * Calculate variance of color values
     */
    private function calculate_variance(array $values): float {
        $count = count($values);
        if ($count < 2) {
            return 0;
        }

        $mean = array_sum($values) / $count;
        $variance = 0;

        foreach ($values as $value) {
            $variance += pow($value - $mean, 2);
        }

        return $variance / $count;
    }

    /**
     * Compress JPEG image
     */
    private function compress_jpeg(string $file_path, string $level): array {
        $quality = self::QUALITY_PRESETS[$level]['jpg'];

        try {
            $image = @imagecreatefromjpeg($file_path);
            if (!$image) {
                return ['success' => false, 'message' => __('Failed to load JPEG image.', 'en-bombas')];
            }

            // Remove EXIF by creating a new image
            if ($this->remove_exif) {
                $width = imagesx($image);
                $height = imagesy($image);
                $new_image = imagecreatetruecolor($width, $height);
                imagecopy($new_image, $image, 0, 0, 0, 0, $width, $height);
                imagedestroy($image);
                $image = $new_image;
            }

            $result = imagejpeg($image, $file_path, $quality);
            imagedestroy($image);

            if (!$result) {
                return ['success' => false, 'message' => __('Failed to save JPEG image.', 'en-bombas')];
            }

            return ['success' => true];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Compress PNG image
     */
    private function compress_png(string $file_path, string $level): array {
        $compression = self::QUALITY_PRESETS[$level]['png'];

        try {
            $image = @imagecreatefrompng($file_path);
            if (!$image) {
                return ['success' => false, 'message' => __('Failed to load PNG image.', 'en-bombas')];
            }

            // Preserve transparency
            imagealphablending($image, false);
            imagesavealpha($image, true);

            $result = imagepng($image, $file_path, $compression);
            imagedestroy($image);

            if (!$result) {
                return ['success' => false, 'message' => __('Failed to save PNG image.', 'en-bombas')];
            }

            return ['success' => true];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Compress GIF image
     */
    private function compress_gif(string $file_path, string $level): array {
        try {
            $image = @imagecreatefromgif($file_path);
            if (!$image) {
                return ['success' => false, 'message' => __('Failed to load GIF image.', 'en-bombas')];
            }

            // Reduce colors based on compression level
            $colors = match ($level) {
                'ultra' => 64,
                'high' => 128,
                default => 256,
            };

            imagetruecolortopalette($image, true, $colors);

            $result = imagegif($image, $file_path);
            imagedestroy($image);

            if (!$result) {
                return ['success' => false, 'message' => __('Failed to save GIF image.', 'en-bombas')];
            }

            return ['success' => true];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Compress PDF file
     */
    private function compress_pdf(string $file_path, string $level): array {
        // Check if Ghostscript is available
        $gs_path = $this->find_ghostscript();
        if (!$gs_path) {
            return [
                'success' => false,
                'message' => __('PDF compression requires Ghostscript to be installed on the server.', 'en-bombas'),
            ];
        }

        $quality_setting = match ($level) {
            'ultra' => '/screen',
            'high' => '/ebook',
            default => '/printer',
        };

        $temp_file = $file_path . '.tmp';

        $command = sprintf(
            '%s -sDEVICE=pdfwrite -dCompatibilityLevel=1.4 -dPDFSETTINGS=%s -dNOPAUSE -dQUIET -dBATCH -sOutputFile=%s %s 2>&1',
            escapeshellcmd($gs_path),
            $quality_setting,
            escapeshellarg($temp_file),
            escapeshellarg($file_path)
        );

        exec($command, $output, $return_code);

        if ($return_code !== 0 || !file_exists($temp_file)) {
            @unlink($temp_file);
            return [
                'success' => false,
                'message' => __('PDF compression failed.', 'en-bombas'),
            ];
        }

        // Replace original with compressed version
        @unlink($file_path);
        rename($temp_file, $file_path);

        return ['success' => true];
    }

    /**
     * Find Ghostscript executable
     */
    private function find_ghostscript(): ?string {
        $possible_paths = [
            'gs',
            '/usr/bin/gs',
            '/usr/local/bin/gs',
            '/opt/local/bin/gs',
        ];

        foreach ($possible_paths as $path) {
            exec("which $path 2>/dev/null", $output, $return_code);
            if ($return_code === 0) {
                return $path;
            }
        }

        return null;
    }

    /**
     * Create WebP version of an image
     */
    public function create_webp(string $source_path, string $output_path, string $level = 'normal'): array {
        if (!function_exists('imagewebp')) {
            return [
                'success' => false,
                'message' => __('WebP support not available in PHP.', 'en-bombas'),
            ];
        }

        $quality = self::QUALITY_PRESETS[$level]['webp'];
        $mime_type = wp_check_filetype($source_path)['type'];

        try {
            $image = match ($mime_type) {
                'image/jpeg', 'image/jpg' => @imagecreatefromjpeg($source_path),
                'image/png' => @imagecreatefrompng($source_path),
                'image/gif' => @imagecreatefromgif($source_path),
                default => null,
            };

            if (!$image) {
                return ['success' => false, 'message' => __('Failed to load source image.', 'en-bombas')];
            }

            // Handle transparency for PNG
            if ($mime_type === 'image/png') {
                imagepalettetotruecolor($image);
                imagealphablending($image, true);
                imagesavealpha($image, true);
            }

            $result = imagewebp($image, $output_path, $quality);
            imagedestroy($image);

            if (!$result) {
                return ['success' => false, 'message' => __('Failed to create WebP image.', 'en-bombas')];
            }

            return [
                'success' => true,
                'path' => $output_path,
                'size' => filesize($output_path),
            ];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Create AVIF version of an image
     */
    public function create_avif(string $source_path, string $output_path, string $level = 'normal'): array {
        if (!function_exists('imageavif')) {
            return [
                'success' => false,
                'message' => __('AVIF support not available in PHP (requires PHP 8.1+).', 'en-bombas'),
            ];
        }

        $quality = self::QUALITY_PRESETS[$level]['avif'];
        $mime_type = wp_check_filetype($source_path)['type'];

        try {
            $image = match ($mime_type) {
                'image/jpeg', 'image/jpg' => @imagecreatefromjpeg($source_path),
                'image/png' => @imagecreatefrompng($source_path),
                'image/gif' => @imagecreatefromgif($source_path),
                default => null,
            };

            if (!$image) {
                return ['success' => false, 'message' => __('Failed to load source image.', 'en-bombas')];
            }

            // Handle transparency for PNG
            if ($mime_type === 'image/png') {
                imagepalettetotruecolor($image);
                imagealphablending($image, true);
                imagesavealpha($image, true);
            }

            $result = imageavif($image, $output_path, $quality);
            imagedestroy($image);

            if (!$result) {
                return ['success' => false, 'message' => __('Failed to create AVIF image.', 'en-bombas')];
            }

            return [
                'success' => true,
                'path' => $output_path,
                'size' => filesize($output_path),
            ];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}



