<?php
/**
 * Bulk Optimizer Class
 *
 * @package EnBombas\Bulk
 */

namespace EnBombas\Bulk;

use EnBombas\Optimizer\ImageOptimizer;
use EnBombas\NextGen\NextGenFormats;
use EnBombas\Core\Plugin;

/**
 * Class BulkOptimizer
 *
 * Handles bulk image optimization.
 */
class BulkOptimizer {
    /**
     * Batch size for processing
     */
    private const BATCH_SIZE = 5;

    /**
     * Turbo mode batch size
     */
    private const TURBO_BATCH_SIZE = 10;

    /**
     * Option key for queue
     */
    private const QUEUE_OPTION = 'enbombas_bulk_queue';

    /**
     * Option key for status
     */
    private const STATUS_OPTION = 'enbombas_bulk_status';

    /**
     * Image optimizer instance
     */
    private ImageOptimizer $optimizer;

    /**
     * Next-gen formats handler
     */
    private NextGenFormats $nextgen;

    /**
     * Constructor
     */
    public function __construct(ImageOptimizer $optimizer) {
        $this->optimizer = $optimizer;
        $this->nextgen = new NextGenFormats();
    }

    /**
     * Start bulk optimization
     */
    public function start(): array {
        // Get unoptimized images
        $unoptimized = $this->optimizer->get_unoptimized_attachments();

        if (empty($unoptimized)) {
            return [
                'success' => true,
                'message' => __('All images are already optimized!', 'en-bombas'),
                'complete' => true,
                'total' => 0,
            ];
        }

        // Initialize queue
        update_option(self::QUEUE_OPTION, $unoptimized);

        // Initialize status
        $status = [
            'running' => true,
            'total' => count($unoptimized),
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'total_saved' => 0,
            'started_at' => current_time('mysql'),
            'errors' => [],
        ];
        update_option(self::STATUS_OPTION, $status);

        return [
            'success' => true,
            'message' => sprintf(__('Starting optimization of %d images...', 'en-bombas'), count($unoptimized)),
            'total' => count($unoptimized),
            'processed' => 0,
        ];
    }

    /**
     * Process a batch of images
     */
    public function process_batch(): array {
        $queue = get_option(self::QUEUE_OPTION, []);
        $status = get_option(self::STATUS_OPTION, []);

        if (empty($queue) || empty($status['running'])) {
            return [
                'success' => true,
                'complete' => true,
                'status' => $status,
            ];
        }

        $plugin = Plugin::get_instance();
        $turbo_mode = $plugin->get_setting('turbo_mode', false);
        $batch_size = $turbo_mode ? self::TURBO_BATCH_SIZE : self::BATCH_SIZE;

        // Get batch to process
        $batch = array_splice($queue, 0, $batch_size);

        // Process each image
        $batch_results = [];
        foreach ($batch as $attachment_id) {
            $result = $this->process_single($attachment_id);
            $batch_results[] = $result;

            // Update status
            $status['processed']++;

            if ($result['success']) {
                $status['success']++;
                $status['total_saved'] += $result['saved_bytes'] ?? 0;
            } else {
                $status['failed']++;
                $status['errors'][] = [
                    'id' => $attachment_id,
                    'message' => $result['message'] ?? __('Unknown error', 'en-bombas'),
                ];
            }
        }

        // Update queue
        update_option(self::QUEUE_OPTION, $queue);

        // Check if complete
        $complete = empty($queue);
        if ($complete) {
            $status['running'] = false;
            $status['completed_at'] = current_time('mysql');
        }

        // Update status
        update_option(self::STATUS_OPTION, $status);

        return [
            'success' => true,
            'complete' => $complete,
            'status' => $status,
            'batch_results' => $batch_results,
            'remaining' => count($queue),
        ];
    }

    /**
     * Process single image
     */
    private function process_single(int $attachment_id): array {
        $plugin = Plugin::get_instance();

        // Optimize the image
        $result = $this->optimizer->optimize($attachment_id);

        if (!$result['success']) {
            return $result;
        }

        // Generate next-gen formats if enabled
        if ($plugin->get_setting('generate_webp')) {
            $this->nextgen->generate_webp($attachment_id);
        }

        if ($plugin->get_setting('generate_avif')) {
            $this->nextgen->generate_avif($attachment_id);
        }

        return $result;
    }

    /**
     * Stop bulk optimization
     */
    public function stop(): array {
        $status = get_option(self::STATUS_OPTION, []);
        $status['running'] = false;
        $status['stopped_at'] = current_time('mysql');
        update_option(self::STATUS_OPTION, $status);

        // Clear queue
        delete_option(self::QUEUE_OPTION);

        return [
            'success' => true,
            'message' => __('Bulk optimization stopped.', 'en-bombas'),
            'status' => $status,
        ];
    }

    /**
     * Get current status
     */
    public function get_status(): array {
        $status = get_option(self::STATUS_OPTION, []);
        $queue = get_option(self::QUEUE_OPTION, []);

        if (empty($status)) {
            return [
                'running' => false,
                'total' => 0,
                'processed' => 0,
                'success' => 0,
                'failed' => 0,
                'total_saved' => 0,
                'remaining' => count($queue),
            ];
        }

        $status['remaining'] = count($queue);
        $status['total_saved_formatted'] = size_format($status['total_saved'] ?? 0);
        $status['progress'] = $status['total'] > 0
            ? round(($status['processed'] / $status['total']) * 100, 1)
            : 0;

        return $status;
    }

    /**
     * Reset bulk optimization status
     */
    public function reset(): void {
        delete_option(self::QUEUE_OPTION);
        delete_option(self::STATUS_OPTION);
    }

    /**
     * Check if bulk optimization is running
     */
    public function is_running(): bool {
        $status = get_option(self::STATUS_OPTION, []);
        return !empty($status['running']);
    }

    /**
     * Get optimization summary
     */
    public function get_summary(): array {
        $plugin = Plugin::get_instance();
        $stats = $plugin->get_statistics();

        return [
            'total_images' => $stats['total_images'],
            'optimized' => $stats['optimized_images'],
            'unoptimized' => $stats['unoptimized_images'],
            'total_saved' => $stats['total_saved_formatted'],
            'optimization_rate' => $stats['optimization_rate'],
        ];
    }

    /**
     * Get images from custom folders
     */
    public function get_custom_folder_images(): array {
        $plugin = Plugin::get_instance();
        $custom_folders = $plugin->get_setting('custom_folders', []);

        if (empty($custom_folders)) {
            return [];
        }

        $images = [];
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif'];

        foreach ($custom_folders as $folder) {
            if (!is_dir($folder)) {
                continue;
            }

            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($folder, \RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                if (!$file->isFile()) {
                    continue;
                }

                $extension = strtolower($file->getExtension());
                if (in_array($extension, $allowed_extensions)) {
                    $images[] = $file->getPathname();
                }
            }
        }

        return $images;
    }

    /**
     * Optimize custom folder images
     */
    public function optimize_custom_folders(): array {
        $images = $this->get_custom_folder_images();

        if (empty($images)) {
            return [
                'success' => true,
                'message' => __('No images found in custom folders.', 'en-bombas'),
                'optimized' => 0,
            ];
        }

        $optimized = 0;
        $total_saved = 0;
        $errors = [];

        foreach ($images as $image_path) {
            $result = $this->optimizer->optimize_file($image_path);

            if ($result['success']) {
                $optimized++;
                $total_saved += $result['saved_bytes'] ?? 0;
            } else {
                $errors[] = [
                    'file' => $image_path,
                    'message' => $result['message'],
                ];
            }
        }

        return [
            'success' => true,
            'optimized' => $optimized,
            'total' => count($images),
            'total_saved' => $total_saved,
            'total_saved_formatted' => size_format($total_saved),
            'errors' => $errors,
        ];
    }
}



