<?php
/**
 * Backup Manager Class
 *
 * @package EnBombas\Backup
 */

namespace EnBombas\Backup;

/**
 * Class BackupManager
 *
 * Handles backup and restore of original images.
 */
class BackupManager {
    /**
     * Backup directory path
     */
    private string $backup_dir;

    /**
     * Constructor
     */
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->backup_dir = $upload_dir['basedir'] . '/enbombas-backups';

        $this->ensure_backup_dir();
    }

    /**
     * Ensure backup directory exists
     */
    private function ensure_backup_dir(): void {
        if (!file_exists($this->backup_dir)) {
            wp_mkdir_p($this->backup_dir);

            // Security files
            file_put_contents($this->backup_dir . '/index.php', '<?php // Silence is golden');
            file_put_contents($this->backup_dir . '/.htaccess', 'deny from all');
        }
    }

    /**
     * Create backup of an attachment
     */
    public function backup(int $attachment_id): array {
        $file_path = get_attached_file($attachment_id);

        if (!$file_path || !file_exists($file_path)) {
            return [
                'success' => false,
                'message' => __('Original file not found.', 'en-bombas'),
            ];
        }

        // Create backup subdirectory based on year/month
        $upload_dir = wp_upload_dir();
        $relative_path = str_replace($upload_dir['basedir'] . '/', '', $file_path);
        $backup_path = $this->backup_dir . '/' . $relative_path;
        $backup_dir = dirname($backup_path);

        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
        }

        // Copy original file
        if (!copy($file_path, $backup_path)) {
            return [
                'success' => false,
                'message' => __('Failed to create backup file.', 'en-bombas'),
            ];
        }

        // Backup thumbnails
        $metadata = wp_get_attachment_metadata($attachment_id);
        $backed_up_thumbs = [];

        if ($metadata && !empty($metadata['sizes'])) {
            $base_dir = dirname($file_path);
            $backup_base_dir = dirname($backup_path);

            foreach ($metadata['sizes'] as $size => $data) {
                $thumb_path = $base_dir . '/' . $data['file'];
                $thumb_backup = $backup_base_dir . '/' . $data['file'];

                if (file_exists($thumb_path)) {
                    if (copy($thumb_path, $thumb_backup)) {
                        $backed_up_thumbs[] = $data['file'];
                    }
                }
            }
        }

        // Save backup info to meta
        update_post_meta($attachment_id, '_enbombas_backup', [
            'path' => $backup_path,
            'original_size' => filesize($backup_path),
            'thumbnails' => $backed_up_thumbs,
            'created_at' => current_time('mysql'),
        ]);

        return [
            'success' => true,
            'message' => __('Backup created successfully.', 'en-bombas'),
            'backup_path' => $backup_path,
        ];
    }

    /**
     * Restore original from backup
     */
    public function restore(int $attachment_id): array {
        $backup_info = get_post_meta($attachment_id, '_enbombas_backup', true);

        if (!$backup_info || empty($backup_info['path'])) {
            return [
                'success' => false,
                'message' => __('No backup found for this image.', 'en-bombas'),
            ];
        }

        $backup_path = $backup_info['path'];
        $file_path = get_attached_file($attachment_id);

        if (!file_exists($backup_path)) {
            return [
                'success' => false,
                'message' => __('Backup file not found.', 'en-bombas'),
            ];
        }

        // Restore main file
        if (!copy($backup_path, $file_path)) {
            return [
                'success' => false,
                'message' => __('Failed to restore original file.', 'en-bombas'),
            ];
        }

        // Restore thumbnails
        if (!empty($backup_info['thumbnails'])) {
            $base_dir = dirname($file_path);
            $backup_base_dir = dirname($backup_path);

            foreach ($backup_info['thumbnails'] as $thumb_file) {
                $thumb_backup = $backup_base_dir . '/' . $thumb_file;
                $thumb_path = $base_dir . '/' . $thumb_file;

                if (file_exists($thumb_backup)) {
                    copy($thumb_backup, $thumb_path);
                }
            }
        }

        // Clear optimization data
        delete_post_meta($attachment_id, '_enbombas_optimized');
        delete_post_meta($attachment_id, '_enbombas_data');
        delete_post_meta($attachment_id, '_enbombas_saved_bytes');

        // Delete next-gen versions
        $this->delete_nextgen_files($attachment_id);

        return [
            'success' => true,
            'message' => __('Original image restored successfully.', 'en-bombas'),
        ];
    }

    /**
     * Delete backup files for an attachment
     */
    public function delete_backup(int $attachment_id): bool {
        $backup_info = get_post_meta($attachment_id, '_enbombas_backup', true);

        if (!$backup_info || empty($backup_info['path'])) {
            return false;
        }

        $backup_path = $backup_info['path'];

        // Delete main backup
        if (file_exists($backup_path)) {
            @unlink($backup_path);
        }

        // Delete thumbnail backups
        if (!empty($backup_info['thumbnails'])) {
            $backup_base_dir = dirname($backup_path);

            foreach ($backup_info['thumbnails'] as $thumb_file) {
                $thumb_backup = $backup_base_dir . '/' . $thumb_file;
                if (file_exists($thumb_backup)) {
                    @unlink($thumb_backup);
                }
            }
        }

        // Remove meta
        delete_post_meta($attachment_id, '_enbombas_backup');

        return true;
    }

    /**
     * Delete next-gen format files
     */
    private function delete_nextgen_files(int $attachment_id): void {
        // WebP files
        $webp_files = get_post_meta($attachment_id, '_enbombas_webp', true);
        if (is_array($webp_files)) {
            foreach ($webp_files as $file) {
                if (file_exists($file)) {
                    @unlink($file);
                }
            }
        }
        delete_post_meta($attachment_id, '_enbombas_webp');

        // AVIF files
        $avif_files = get_post_meta($attachment_id, '_enbombas_avif', true);
        if (is_array($avif_files)) {
            foreach ($avif_files as $file) {
                if (file_exists($file)) {
                    @unlink($file);
                }
            }
        }
        delete_post_meta($attachment_id, '_enbombas_avif');
    }

    /**
     * Check if backup exists for an attachment
     */
    public function has_backup(int $attachment_id): bool {
        $backup_info = get_post_meta($attachment_id, '_enbombas_backup', true);

        if (!$backup_info || empty($backup_info['path'])) {
            return false;
        }

        return file_exists($backup_info['path']);
    }

    /**
     * Get backup info for an attachment
     */
    public function get_backup_info(int $attachment_id): ?array {
        $info = get_post_meta($attachment_id, '_enbombas_backup', true);
        return is_array($info) ? $info : null;
    }

    /**
     * Get total backup size
     */
    public function get_total_backup_size(): int {
        return $this->get_directory_size($this->backup_dir);
    }

    /**
     * Get directory size recursively
     */
    private function get_directory_size(string $dir): int {
        $size = 0;

        if (!is_dir($dir)) {
            return $size;
        }

        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($dir, \RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }

        return $size;
    }

    /**
     * Clear all backups
     */
    public function clear_all_backups(): array {
        global $wpdb;

        // Get all attachment IDs with backups
        $attachment_ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s",
                '_enbombas_backup'
            )
        );

        $deleted = 0;
        foreach ($attachment_ids as $id) {
            if ($this->delete_backup((int) $id)) {
                $deleted++;
            }
        }

        // Clean up empty directories
        $this->cleanup_empty_dirs($this->backup_dir);

        return [
            'success' => true,
            'deleted' => $deleted,
            'message' => sprintf(__('Deleted %d backup files.', 'en-bombas'), $deleted),
        ];
    }

    /**
     * Remove empty directories recursively
     */
    private function cleanup_empty_dirs(string $dir): bool {
        if (!is_dir($dir)) {
            return false;
        }

        $files = array_diff(scandir($dir), ['.', '..']);

        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->cleanup_empty_dirs($path);
            }
        }

        // Re-check if empty after cleaning subdirs
        $files = array_diff(scandir($dir), ['.', '..']);

        // Don't delete the main backup dir
        if (empty($files) && $dir !== $this->backup_dir) {
            return @rmdir($dir);
        }

        return true;
    }

    /**
     * Get backup directory path
     */
    public function get_backup_dir(): string {
        return $this->backup_dir;
    }
}



