<?php
/**
 * REST API Handler
 *
 * @package EnBombas\AdminUI
 */

namespace EnBombas\AdminUI;

/**
 * Class RestAPI
 *
 * Handles REST API endpoints for En Bombas.
 */
class RestAPI {
    /**
     * Constructor
     */
    public function __construct() {
        add_action('rest_api_init', [$this, 'register_routes']);
        add_action('wp_ajax_enbombas_update_setting', [$this, 'handle_ajax_update_setting']);
    }

    /**
     * Register REST routes
     */
    public function register_routes(): void {
        register_rest_route('enbombas/v1', '/update-setting', [
            'methods' => 'POST',
            'callback' => [$this, 'update_setting_rest'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            },
        ]);
    }

    /**
     * REST endpoint: Update setting
     */
    public function update_setting_rest(\WP_REST_Request $request): \WP_REST_Response {
        $key = sanitize_text_field($request->get_param('key'));
        $value = $request->get_param('value');

        if (empty($key)) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => __('Invalid key', 'en-bombas'),
            ], 400);
        }

        // Handle Nitro settings (stored in array under 'enbombas_nitro_settings')
        $nitro = \EnBombas\Nitro\NitroCore::get_instance();
        $settings = $nitro->get_settings();
        
        // Check if key has prefix or is a direct Nitro setting
        $setting_key = $key;
        if (strpos($key, 'enbombas_nitro_') === 0) {
            $setting_key = str_replace('enbombas_nitro_', '', $key);
        }
        
        // Check if this is a Nitro setting
        if (isset($settings[$setting_key])) {
            // Convert value to boolean/int as needed
            if ($value === '1' || $value === '0') {
                $settings[$setting_key] = (bool) $value;
            } elseif (is_numeric($value)) {
                $settings[$setting_key] = (int) $value;
            } else {
                $settings[$setting_key] = $value;
            }
            $nitro->update_settings($settings);
        } else {
            // Direct option update for non-Nitro settings
            update_option($key, $value);
        }

        return new \WP_REST_Response([
            'success' => true,
            'key' => $key,
            'value' => $value,
        ], 200);
    }

    /**
     * AJAX fallback: Update setting
     */
    public function handle_ajax_update_setting(): void {
        // Check nonce - can be from either enbombas_nonce or wp_rest
        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
        if (!wp_verify_nonce($nonce, 'enbombas_nonce') && !wp_verify_nonce($nonce, 'wp_rest')) {
            wp_send_json_error(['message' => __('Invalid nonce', 'en-bombas')]);
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('No tienes permiso para hacer esto, parce.', 'en-bombas')]);
        }

        $key = isset($_POST['key']) ? sanitize_text_field($_POST['key']) : '';
        $value = isset($_POST['value']) ? sanitize_text_field($_POST['value']) : '';

        if (empty($key)) {
            wp_send_json_error(['message' => __('Invalid key', 'en-bombas')]);
        }

        // Handle Nitro settings (stored in array under 'enbombas_nitro_settings')
        $nitro = \EnBombas\Nitro\NitroCore::get_instance();
        $settings = $nitro->get_settings();
        
        // Check if key has prefix or is a direct Nitro setting
        $setting_key = $key;
        if (strpos($key, 'enbombas_nitro_') === 0) {
            $setting_key = str_replace('enbombas_nitro_', '', $key);
        }
        
        // Check if this is a Nitro setting
        if (isset($settings[$setting_key])) {
            // Convert value to boolean/int as needed
            if ($value === '1' || $value === '0') {
                $settings[$setting_key] = (bool) $value;
            } elseif (is_numeric($value)) {
                $settings[$setting_key] = (int) $value;
            } else {
                $settings[$setting_key] = $value;
            }
            $nitro->update_settings($settings);
        } else {
            // Direct option update for non-Nitro settings
            update_option($key, $value);
        }

        wp_send_json_success([
            'key' => $key,
            'value' => $value,
        ]);
    }
}

