<?php
/**
 * Media Library Integration Class
 *
 * @package EnBombas\AdminUI
 */

namespace EnBombas\AdminUI;

use EnBombas\Optimizer\ImageOptimizer;
use EnBombas\Backup\BackupManager;

/**
 * Class MediaLibrary
 *
 * Integrates En Bombas with WordPress Media Library.
 */
class MediaLibrary {
    /**
     * Image optimizer instance
     */
    private ImageOptimizer $optimizer;

    /**
     * Backup manager instance
     */
    private BackupManager $backup_manager;

    /**
     * Constructor
     */
    public function __construct(ImageOptimizer $optimizer, BackupManager $backup_manager) {
        $this->optimizer = $optimizer;
        $this->backup_manager = $backup_manager;

        // Add column to Media Library
        add_filter('manage_media_columns', [$this, 'add_column']);
        add_action('manage_media_custom_column', [$this, 'render_column'], 10, 2);

        // Add attachment fields
        add_filter('attachment_fields_to_edit', [$this, 'add_attachment_fields'], 10, 2);

        // Hook into attachment deletion
        add_action('delete_attachment', [$this, 'cleanup_on_delete']);
    }

    /**
     * Add En Bombas Status column
     */
    public function add_column(array $columns): array {
        $columns['enbombas_status'] = __('En Bombas Status', 'en-bombas');
        return $columns;
    }

    /**
     * Render En Bombas Status column
     */
    public function render_column(string $column_name, int $attachment_id): void {
        if ($column_name !== 'enbombas_status') {
            return;
        }

        // Check if it's an optimizable image
        if (!wp_attachment_is_image($attachment_id)) {
            $mime = get_post_mime_type($attachment_id);
            if ($mime !== 'application/pdf') {
                echo '<span class="enbombas-status-na">' . esc_html__('N/A', 'en-bombas') . '</span>';
                return;
            }
        }

        $is_optimized = $this->optimizer->is_optimized($attachment_id);
        $data = $this->optimizer->get_optimization_data($attachment_id);
        $has_backup = $this->backup_manager->has_backup($attachment_id);

        echo '<div class="enbombas-media-status" data-attachment-id="' . esc_attr($attachment_id) . '">';

        if ($is_optimized && $data) {
            // Optimized state
            echo '<div class="enbombas-status enbombas-status-optimized">';
            echo '<span class="enbombas-status-icon"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"/></svg></span>';
            echo '<span class="enbombas-status-text">' . esc_html__('Optimized', 'en-bombas') . '</span>';
            echo '</div>';

            // Savings info
            echo '<div class="enbombas-savings">';
            echo '<span class="enbombas-saved-badge">';
            echo sprintf(
                esc_html__('-%s%%', 'en-bombas'),
                esc_html($data['saved_percentage'])
            );
            echo '</span>';
            echo '<span class="enbombas-saved-size">';
            echo esc_html(size_format($data['saved_bytes']));
            echo '</span>';
            echo '</div>';

            // Compression level
            echo '<div class="enbombas-level">';
            echo '<span class="enbombas-level-badge enbombas-level-' . esc_attr($data['compression_level']) . '">';
            echo esc_html(ucfirst($data['compression_level']));
            echo '</span>';
            echo '</div>';

            // Actions
            echo '<div class="enbombas-actions">';
            echo '<button type="button" class="button button-small enbombas-reoptimize" data-id="' . esc_attr($attachment_id) . '">';
            echo esc_html__('Re-optimize', 'en-bombas');
            echo '</button>';

            if ($has_backup) {
                echo '<button type="button" class="button button-small enbombas-restore" data-id="' . esc_attr($attachment_id) . '">';
                echo esc_html__('Restore', 'en-bombas');
                echo '</button>';
            }
            echo '</div>';
        } else {
            // Not optimized state
            echo '<div class="enbombas-status enbombas-status-unoptimized">';
            echo '<span class="enbombas-status-icon"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><circle cx="12" cy="12" r="10"/><line x1="12" y1="8" x2="12" y2="12"/><line x1="12" y1="16" x2="12.01" y2="16"/></svg></span>';
            echo '<span class="enbombas-status-text">' . esc_html__('Not optimized', 'en-bombas') . '</span>';
            echo '</div>';

            // Action
            echo '<div class="enbombas-actions">';
            echo '<button type="button" class="button button-primary button-small enbombas-optimize" data-id="' . esc_attr($attachment_id) . '">';
            echo '<span class="enbombas-btn-text">' . esc_html__('Optimize', 'en-bombas') . '</span>';
            echo '<span class="enbombas-btn-loader" style="display:none;"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2" fill="none" stroke-dasharray="31.416" stroke-dashoffset="31.416"><animate attributeName="stroke-dasharray" dur="2s" values="0 31.416;15.708 15.708;0 31.416;0 31.416" repeatCount="indefinite"/><animate attributeName="stroke-dashoffset" dur="2s" values="0;-15.708;-31.416;-31.416" repeatCount="indefinite"/></circle></svg></span>';
            echo '</button>';
            echo '</div>';
        }

        echo '</div>';
    }

    /**
     * Add En Bombas fields to attachment edit screen
     */
    public function add_attachment_fields(array $form_fields, \WP_Post $post): array {
        if (!wp_attachment_is_image($post->ID)) {
            $mime = get_post_mime_type($post->ID);
            if ($mime !== 'application/pdf') {
                return $form_fields;
            }
        }

        $is_optimized = $this->optimizer->is_optimized($post->ID);
        $data = $this->optimizer->get_optimization_data($post->ID);
        $has_backup = $this->backup_manager->has_backup($post->ID);

        ob_start();
        ?>
        <div class="enbombas-attachment-field">
            <?php if ($is_optimized && $data): ?>
                <div class="enbombas-field-status enbombas-optimized">
                    <strong><?php esc_html_e('Status:', 'en-bombas'); ?></strong>
                    <span class="status-optimized"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14" style="vertical-align:middle;margin-right:4px;"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"/></svg> <?php esc_html_e('Optimized', 'en-bombas'); ?></span>
                </div>
                <div class="enbombas-field-details">
                    <p>
                        <strong><?php esc_html_e('Saved:', 'en-bombas'); ?></strong>
                        <?php echo esc_html(size_format($data['saved_bytes'])); ?>
                        (<?php echo esc_html($data['saved_percentage']); ?>%)
                    </p>
                    <p>
                        <strong><?php esc_html_e('Level:', 'en-bombas'); ?></strong>
                        <?php echo esc_html(ucfirst($data['compression_level'])); ?>
                    </p>
                    <p>
                        <strong><?php esc_html_e('Optimized:', 'en-bombas'); ?></strong>
                        <?php echo esc_html($data['optimized_at']); ?>
                    </p>
                </div>
                <div class="enbombas-field-actions">
                    <button type="button" class="button enbombas-reoptimize" data-id="<?php echo esc_attr($post->ID); ?>">
                        <?php esc_html_e('Re-optimize', 'en-bombas'); ?>
                    </button>
                    <?php if ($has_backup): ?>
                        <button type="button" class="button enbombas-restore" data-id="<?php echo esc_attr($post->ID); ?>">
                            <?php esc_html_e('Restore Original', 'en-bombas'); ?>
                        </button>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <div class="enbombas-field-status enbombas-not-optimized">
                    <strong><?php esc_html_e('Status:', 'en-bombas'); ?></strong>
                    <span class="status-unoptimized">🔴 <?php esc_html_e('Not optimized', 'en-bombas'); ?></span>
                </div>
                <div class="enbombas-field-actions">
                    <button type="button" class="button button-primary enbombas-optimize" data-id="<?php echo esc_attr($post->ID); ?>">
                        <?php esc_html_e('Optimize Now', 'en-bombas'); ?>
                    </button>
                </div>
            <?php endif; ?>
        </div>
        <?php
        $html = ob_get_clean();

        $form_fields['enbombas'] = [
            'label' => __('En Bombas', 'en-bombas'),
            'input' => 'html',
            'html' => $html,
        ];

        return $form_fields;
    }

    /**
     * Cleanup on attachment deletion
     */
    public function cleanup_on_delete(int $attachment_id): void {
        // Delete backup
        $this->backup_manager->delete_backup($attachment_id);

        // Delete next-gen files
        $webp_files = get_post_meta($attachment_id, '_enbombas_webp', true);
        if (is_array($webp_files)) {
            foreach ($webp_files as $file) {
                if (file_exists($file)) {
                    @unlink($file);
                }
            }
        }

        $avif_files = get_post_meta($attachment_id, '_enbombas_avif', true);
        if (is_array($avif_files)) {
            foreach ($avif_files as $file) {
                if (file_exists($file)) {
                    @unlink($file);
                }
            }
        }

        // Clean up all meta
        delete_post_meta($attachment_id, '_enbombas_optimized');
        delete_post_meta($attachment_id, '_enbombas_data');
        delete_post_meta($attachment_id, '_enbombas_saved_bytes');
        delete_post_meta($attachment_id, '_enbombas_backup');
        delete_post_meta($attachment_id, '_enbombas_webp');
        delete_post_meta($attachment_id, '_enbombas_avif');
    }
}



