<?php
/**
 * Admin Pages Class
 *
 * @package EnBombas\AdminUI
 */

namespace EnBombas\AdminUI;

use EnBombas\Core\Plugin;

/**
 * Class AdminPages
 *
 * Handles admin menu pages and settings.
 */
class AdminPages {
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', [$this, 'register_menus']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_filter('admin_body_class', [$this, 'add_admin_body_class']);
    }

    /**
     * Add body class to En Bombas pages
     */
    public function add_admin_body_class(string $classes): string {
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'en-bombas') !== false) {
            $classes .= ' enbombas-page';
        }
        return $classes;
    }

    /**
     * Register admin menus
     */
    public function register_menus(): void {
        // Main menu
        add_menu_page(
            __('En Bombas', 'en-bombas'),
            __('En Bombas', 'en-bombas'),
            'upload_files',
            'en-bombas',
            [$this, 'render_dashboard'],
            $this->get_menu_icon(),
            30
        );

        // Dashboard submenu
        add_submenu_page(
            'en-bombas',
            __('Panel de Control', 'en-bombas'),
            __('Panel de Control', 'en-bombas'),
            'upload_files',
            'en-bombas',
            [$this, 'render_dashboard']
        );

        // Bulk Optimizer
        add_submenu_page(
            'en-bombas',
            __('Optimizador Masivo', 'en-bombas'),
            __('Optimizador Masivo', 'en-bombas'),
            'upload_files',
            'en-bombas-bulk',
            [$this, 'render_bulk_optimizer']
        );

        // Custom Folders
        add_submenu_page(
            'en-bombas',
            __('Carpetas Personalizadas', 'en-bombas'),
            __('Carpetas Personalizadas', 'en-bombas'),
            'manage_options',
            'en-bombas-folders',
            [$this, 'render_custom_folders']
        );

        // Nitro - Performance Module
        add_submenu_page(
            'en-bombas',
            __('Nitro – Rendimiento', 'en-bombas'),
            __('Nitro', 'en-bombas'),
            'manage_options',
            'en-bombas-nitro',
            [$this, 'render_nitro']
        );

        // Settings
        add_submenu_page(
            'en-bombas',
            __('Configuración', 'en-bombas'),
            __('Configuración', 'en-bombas'),
            'manage_options',
            'en-bombas-settings',
            [$this, 'render_settings']
        );
    }

    /**
     * Get SVG menu icon
     */
    private function get_menu_icon(): string {
        // Simplified rocket icon using fill only (no stroke) to match WordPress default menu icons
        // 20x20 viewBox, single flat color via fill="currentColor"
        $svg = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor"><path d="M10 2L3 7v2l7 5 7-5V7l-7-5zm0 8.5L5 12v3l5 2.5L15 15v-3l-5-1.5z"/></svg>';
        return 'data:image/svg+xml;base64,' . base64_encode($svg);
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_assets(string $hook): void {
        // Only load on plugin pages
        if (strpos($hook, 'en-bombas') === false && $hook !== 'upload.php') {
            return;
        }

        // CSS
        wp_enqueue_style(
            'enbombas-admin',
            ENBOMBAS_PLUGIN_URL . 'assets/css/admin.css',
            [],
            ENBOMBAS_VERSION
        );

        // JS
        wp_enqueue_script(
            'enbombas-admin',
            ENBOMBAS_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery'],
            ENBOMBAS_VERSION,
            true
        );

        // Localize script with Spanish strings
        wp_localize_script('enbombas-admin', 'enbombasAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('enbombas_nonce'),
            'strings' => [
                'optimizing' => __('Optimizando...', 'en-bombas'),
                'restoring' => __('Restaurando...', 'en-bombas'),
                'optimized' => __('¡Optimizado!', 'en-bombas'),
                'restored' => __('¡Restaurado!', 'en-bombas'),
                'error' => __('Ocurrió un error', 'en-bombas'),
                'confirm_restore' => __('¿Seguro que querés restaurar la imagen original?', 'en-bombas'),
                'bulk_starting' => __('Iniciando optimización masiva...', 'en-bombas'),
                'bulk_processing' => __('Optimizando imágenes...', 'en-bombas'),
                'bulk_complete' => __('Optimización completada', 'en-bombas'),
                'bulk_stopped' => __('Optimización detenida.', 'en-bombas'),
            ],
        ]);

        // Nitro-specific assets - load on all En Bombas pages
        wp_enqueue_script(
            'enbombas-nitro',
            ENBOMBAS_PLUGIN_URL . 'assets/js/nitro.js',
            ['jquery', 'enbombas-admin'],
            ENBOMBAS_VERSION,
            true
        );

        // Localize with REST API and AJAX data
        wp_localize_script('enbombas-nitro', 'enbData', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'restUrl' => esc_url_raw(rest_url('enbombas/v1/update-setting')),
            'nonce' => wp_create_nonce('wp_rest'),
            'ajaxNonce' => wp_create_nonce('enbombas_nonce'),
            'strings' => [
                'saving' => __('Guardando...', 'en-bombas'),
                'clearing' => __('Limpiando...', 'en-bombas'),
                'optimizing' => __('Optimizando...', 'en-bombas'),
                'saved' => __('¡Guardado!', 'en-bombas'),
                'error' => __('Ocurrió un error', 'en-bombas'),
            ],
        ]);

        // Also localize for nitro admin (backward compatibility)
        wp_localize_script('enbombas-nitro', 'enbombasNitroAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('enbombas_nonce'),
            'strings' => [
                'saving' => __('Guardando...', 'en-bombas'),
                'clearing' => __('Limpiando...', 'en-bombas'),
                'optimizing' => __('Optimizando...', 'en-bombas'),
                'saved' => __('¡Guardado!', 'en-bombas'),
                'error' => __('Ocurrió un error', 'en-bombas'),
            ],
        ]);
    }

    /**
     * Render dashboard page
     */
    public function render_dashboard(): void {
        $plugin = Plugin::get_instance();
        $stats = $plugin->get_statistics();
        $settings = $plugin->get_settings();
        $monitor = $plugin->get_ai_monitor();
        $monitoring_data = $monitor->get_monitoring_data();

        include ENBOMBAS_PLUGIN_DIR . 'templates/dashboard.php';
    }

    /**
     * Render bulk optimizer page
     */
    public function render_bulk_optimizer(): void {
        $plugin = Plugin::get_instance();
        $optimizer = $plugin->get_optimizer();
        $unoptimized_count = $optimizer->get_unoptimized_count();
        $settings = $plugin->get_settings();

        include ENBOMBAS_PLUGIN_DIR . 'templates/bulk-optimizer.php';
    }

    /**
     * Render custom folders page
     */
    public function render_custom_folders(): void {
        $plugin = Plugin::get_instance();
        $custom_folders = $plugin->get_setting('custom_folders', []);

        include ENBOMBAS_PLUGIN_DIR . 'templates/custom-folders.php';
    }

    /**
     * Render settings page
     */
    public function render_settings(): void {
        $plugin = Plugin::get_instance();
        $settings = $plugin->get_settings();

        include ENBOMBAS_PLUGIN_DIR . 'templates/settings.php';
    }

    /**
     * Render Nitro performance page
     */
    public function render_nitro(): void {
        // Initialize Nitro if not already
        $nitro = \EnBombas\Nitro\NitroCore::get_instance();
        $nitro->init();

        include ENBOMBAS_PLUGIN_DIR . 'templates/nitro.php';
    }
}
