<?php
/**
 * Admin Bar Class
 *
 * @package EnBombas\AdminUI
 */

namespace EnBombas\AdminUI;

use EnBombas\Core\Plugin;
use EnBombas\Nitro\NitroCore;

/**
 * Class AdminBar
 *
 * Handles WordPress admin bar menu and quick actions.
 */
class AdminBar {
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_bar_menu', [$this, 'add_admin_bar_menu'], 100);
        add_action('admin_post_enb_clear_cache_all', [$this, 'handle_clear_cache_all']);
        add_action('admin_post_enb_clear_cache_current', [$this, 'handle_clear_cache_current']);
        add_action('admin_post_enb_optimize_images_bg', [$this, 'handle_optimize_images_bg']);
        add_action('admin_notices', [$this, 'show_admin_notices']);
        add_action('wp_head', [$this, 'add_admin_bar_styles']);
        add_action('admin_head', [$this, 'add_admin_bar_styles']);
    }

    /**
     * Add admin bar menu
     */
    public function add_admin_bar_menu(\WP_Admin_Bar $wp_admin_bar): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Main En Bombas menu
        $wp_admin_bar->add_node([
            'id' => 'enbombas',
            'title' => $this->get_menu_icon() . '<span class="ab-label">' . __('En Bombas', 'en-bombas') . '</span>',
            'href' => admin_url('admin.php?page=en-bombas'),
            'meta' => [
                'class' => 'enb-admin-bar-menu',
            ],
        ]);

        // Dashboard link
        $wp_admin_bar->add_node([
            'id' => 'enbombas-dashboard',
            'parent' => 'enbombas',
            'title' => __('Panel de Control', 'en-bombas'),
            'href' => admin_url('admin.php?page=en-bombas'),
        ]);

        // Separator
        $wp_admin_bar->add_node([
            'id' => 'enbombas-sep1',
            'parent' => 'enbombas',
            'title' => '',
            'meta' => [
                'class' => 'enb-admin-bar-separator',
            ],
        ]);

        // Clear all cache
        $wp_admin_bar->add_node([
            'id' => 'enbombas-clear-all',
            'parent' => 'enbombas',
            'title' => '<span class="enb-ab-icon"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><path d="M3 6h18M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6M8 6V4c0-1 1-2 2-2h4c1 0 2 1 2 2v2M10 11v6M14 11v6"/></svg></span>' . __('Vaciar caché de En Bombas', 'en-bombas'),
            'href' => wp_nonce_url(
                admin_url('admin-post.php?action=enb_clear_cache_all'),
                'enb_clear_cache_all'
            ),
        ]);

        // Clear current page cache (only on frontend)
        if (!is_admin()) {
            $current_url = $this->get_current_url();
            $wp_admin_bar->add_node([
                'id' => 'enbombas-clear-current',
                'parent' => 'enbombas',
                'title' => '<span class="enb-ab-icon"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2zM14 2v6h6M10 11v6M14 11v6"/></svg></span>' . __('Vaciar caché de esta página', 'en-bombas'),
                'href' => wp_nonce_url(
                    admin_url('admin-post.php?action=enb_clear_cache_current&url=' . urlencode($current_url)),
                    'enb_clear_cache_current'
                ),
            ]);
        }

        // Separator
        $wp_admin_bar->add_node([
            'id' => 'enbombas-sep2',
            'parent' => 'enbombas',
            'title' => '',
            'meta' => [
                'class' => 'enb-admin-bar-separator',
            ],
        ]);

        // Optimize images in background
        $wp_admin_bar->add_node([
            'id' => 'enbombas-optimize-images',
            'parent' => 'enbombas',
            'title' => '<span class="enb-ab-icon"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><rect x="3" y="3" width="18" height="18" rx="2"/><circle cx="8.5" cy="8.5" r="1.5"/><path d="M21 15l-5-5L5 21"/></svg></span>' . __('Optimizar imágenes en segundo plano', 'en-bombas'),
            'href' => wp_nonce_url(
                admin_url('admin-post.php?action=enb_optimize_images_bg'),
                'enb_optimize_images_bg'
            ),
        ]);

        // Quick links
        $wp_admin_bar->add_node([
            'id' => 'enbombas-sep3',
            'parent' => 'enbombas',
            'title' => '',
            'meta' => [
                'class' => 'enb-admin-bar-separator',
            ],
        ]);

        $wp_admin_bar->add_node([
            'id' => 'enbombas-nitro',
            'parent' => 'enbombas',
            'title' => '<span class="enb-ab-icon"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"/></svg></span>' . __('Nitro – Rendimiento', 'en-bombas'),
            'href' => admin_url('admin.php?page=en-bombas-nitro'),
        ]);

        $wp_admin_bar->add_node([
            'id' => 'enbombas-settings',
            'parent' => 'enbombas',
            'title' => '<span class="enb-ab-icon"><svg viewBox="0 0 24 24" fill="currentColor" width="14" height="14"><circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/></svg></span>' . __('Configuración', 'en-bombas'),
            'href' => admin_url('admin.php?page=en-bombas-settings'),
        ]);
    }

    /**
     * Get menu icon SVG
     */
    private function get_menu_icon(): string {
        return '<span class="enb-ab-rocket"><svg viewBox="0 0 24 24" fill="currentColor" width="16" height="16"><path d="M4.5 16.5c-1.5 1.26-2 5-2 5s3.74-.5 5-2c.71-.84.7-2.13-.09-2.91a2.18 2.18 0 0 0-2.91-.09z"/><path d="m12 15-3-3a22 22 0 0 1 2-3.95A12.88 12.88 0 0 1 22 2c0 2.72-.78 7.5-6 11a22.35 22.35 0 0 1-4 2z"/><path d="M9 12H4s.55-3.03 2-4c1.62-1.08 5 0 5 0"/><path d="M12 15v5s3.03-.55 4-2c1.08-1.62 0-5 0-5"/></svg></span>';
    }

    /**
     * Get current URL
     */
    private function get_current_url(): string {
        $protocol = is_ssl() ? 'https://' : 'http://';
        return $protocol . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
    }

    /**
     * Handle clear all cache action
     */
    public function handle_clear_cache_all(): void {
        if (!current_user_can('manage_options')) {
            wp_die(__('No tienes permiso para hacer esto, parce.', 'en-bombas'));
        }

        check_admin_referer('enb_clear_cache_all');

        $nitro = NitroCore::get_instance();
        $nitro->init();
        $result = $nitro->get_cache_system()->clear_all_cache();

        // Also clear CSS/JS cache
        $nitro->get_css_optimizer()->clear_cache();
        $nitro->get_js_optimizer()->clear_cache();

        // Set transient for admin notice
        set_transient('enb_admin_notice', [
            'type' => 'success',
            'message' => sprintf(
                __('¡Listo, parce! Se limpiaron %d archivos de caché.', 'en-bombas'),
                $result['files_deleted']
            ),
        ], 30);

        // Redirect back
        $redirect_url = wp_get_referer() ?: admin_url('admin.php?page=en-bombas');
        wp_safe_redirect($redirect_url);
        exit;
    }

    /**
     * Handle clear current page cache action
     */
    public function handle_clear_cache_current(): void {
        if (!current_user_can('manage_options')) {
            wp_die(__('No tienes permiso para hacer esto, parce.', 'en-bombas'));
        }

        check_admin_referer('enb_clear_cache_current');

        $url = isset($_GET['url']) ? esc_url_raw(urldecode($_GET['url'])) : '';

        if (!empty($url)) {
            $nitro = NitroCore::get_instance();
            $nitro->init();
            $cleared = $nitro->get_cache_system()->clear_url_cache($url);

            if ($cleared) {
                set_transient('enb_admin_notice', [
                    'type' => 'success',
                    'message' => __('¡Caché de esta página limpiada, vea pues!', 'en-bombas'),
                ], 30);
            } else {
                set_transient('enb_admin_notice', [
                    'type' => 'info',
                    'message' => __('Esta página no estaba en caché.', 'en-bombas'),
                ], 30);
            }
        }

        // Redirect back to the original page
        $redirect_url = !empty($url) ? $url : (wp_get_referer() ?: home_url());
        wp_safe_redirect($redirect_url);
        exit;
    }

    /**
     * Handle optimize images in background action
     */
    public function handle_optimize_images_bg(): void {
        if (!current_user_can('manage_options')) {
            wp_die(__('No tienes permiso para hacer esto, parce.', 'en-bombas'));
        }

        check_admin_referer('enb_optimize_images_bg');

        $plugin = Plugin::get_instance();
        $bulk_optimizer = $plugin->get_bulk_optimizer();

        // Start bulk optimization
        $result = $bulk_optimizer->start();

        if ($result['success']) {
            set_transient('enb_admin_notice', [
                'type' => 'success',
                'message' => sprintf(
                    __('¡Optimización iniciada! %d imágenes en cola. Ve al Optimizador Masivo para ver el progreso.', 'en-bombas'),
                    $result['total'] ?? 0
                ),
            ], 30);
        } else {
            set_transient('enb_admin_notice', [
                'type' => 'warning',
                'message' => $result['message'] ?? __('No hay imágenes pendientes por optimizar.', 'en-bombas'),
            ], 30);
        }

        // Redirect to bulk optimizer
        wp_safe_redirect(admin_url('admin.php?page=en-bombas-bulk'));
        exit;
    }

    /**
     * Show admin notices
     */
    public function show_admin_notices(): void {
        $notice = get_transient('enb_admin_notice');

        if ($notice) {
            delete_transient('enb_admin_notice');

            $type = $notice['type'] ?? 'info';
            $message = $notice['message'] ?? '';

            if (!empty($message)) {
                $class = 'notice notice-' . esc_attr($type) . ' is-dismissible';
                printf(
                    '<div class="%1$s"><p><strong>En Bombas:</strong> %2$s</p></div>',
                    esc_attr($class),
                    esc_html($message)
                );
            }
        }
    }

    /**
     * Add admin bar styles
     */
    public function add_admin_bar_styles(): void {
        if (!is_admin_bar_showing()) {
            return;
        }
        ?>
        <style>
            #wpadminbar .enb-admin-bar-menu > .ab-item {
                display: flex !important;
                align-items: center !important;
                gap: 6px !important;
            }
            #wpadminbar .enb-ab-rocket {
                font-size: 16px;
                line-height: 1;
            }
            #wpadminbar .enb-ab-icon {
                display: inline-block;
                margin-right: 6px;
                font-size: 14px;
            }
            #wpadminbar .enb-admin-bar-separator {
                height: 1px !important;
                background: rgba(255,255,255,0.1) !important;
                margin: 5px 12px !important;
            }
            #wpadminbar .enb-admin-bar-separator .ab-item {
                height: 0 !important;
                padding: 0 !important;
            }
            #wpadminbar #wp-admin-bar-enbombas .ab-sub-wrapper {
                min-width: 260px;
            }
            #wpadminbar #wp-admin-bar-enbombas:hover > .ab-item,
            #wpadminbar #wp-admin-bar-enbombas.hover > .ab-item {
                background: linear-gradient(135deg, #5B3DE6, #7A57D1) !important;
            }
        </style>
        <?php
    }
}



