/**
 * En Bombas Nitro - Frontend JavaScript
 * Handles lazy loading and delayed JS execution
 */
(function() {
    'use strict';

    const config = window.enbombasNitro || {
        lazyload: {
            threshold: 300,
            images: true,
            iframes: true,
            youtube: false
        },
        delayJS: {
            enabled: false,
            timeout: 5000
        }
    };

    /**
     * LazyLoad Module
     */
    const LazyLoad = {
        observer: null,
        loaded: new Set(),

        init: function() {
            if (!config.lazyload.images && !config.lazyload.iframes) {
                return;
            }

            // Use IntersectionObserver if available
            if ('IntersectionObserver' in window) {
                this.initIntersectionObserver();
            } else {
                // Fallback to scroll event
                this.initScrollFallback();
            }

            // Initial check for elements in viewport
            this.checkVisibleElements();
        },

        initIntersectionObserver: function() {
            const options = {
                root: null,
                rootMargin: config.lazyload.threshold + 'px',
                threshold: 0
            };

            this.observer = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        LazyLoad.loadElement(entry.target);
                        LazyLoad.observer.unobserve(entry.target);
                    }
                });
            }, options);

            // Observe lazy elements
            this.observeElements();
        },

        observeElements: function() {
            // Images
            if (config.lazyload.images) {
                document.querySelectorAll('img.enbombas-lazy, img[data-src]').forEach(function(el) {
                    if (!LazyLoad.loaded.has(el)) {
                        LazyLoad.observer.observe(el);
                    }
                });
            }

            // Iframes
            if (config.lazyload.iframes) {
                document.querySelectorAll('.enbombas-lazy-iframe, iframe[data-src]').forEach(function(el) {
                    if (!LazyLoad.loaded.has(el)) {
                        LazyLoad.observer.observe(el);
                    }
                });
            }
        },

        initScrollFallback: function() {
            let ticking = false;
            const self = this;

            function checkScroll() {
                if (!ticking) {
                    requestAnimationFrame(function() {
                        self.checkVisibleElements();
                        ticking = false;
                    });
                    ticking = true;
                }
            }

            window.addEventListener('scroll', checkScroll, { passive: true });
            window.addEventListener('resize', checkScroll, { passive: true });
            window.addEventListener('orientationchange', checkScroll, { passive: true });
        },

        checkVisibleElements: function() {
            const viewportHeight = window.innerHeight || document.documentElement.clientHeight;
            const threshold = config.lazyload.threshold;

            // Check images
            if (config.lazyload.images) {
                document.querySelectorAll('img.enbombas-lazy, img[data-src]').forEach(function(el) {
                    if (!LazyLoad.loaded.has(el) && LazyLoad.isInViewport(el, viewportHeight, threshold)) {
                        LazyLoad.loadElement(el);
                    }
                });
            }

            // Check iframes
            if (config.lazyload.iframes) {
                document.querySelectorAll('.enbombas-lazy-iframe, iframe[data-src]').forEach(function(el) {
                    if (!LazyLoad.loaded.has(el) && LazyLoad.isInViewport(el, viewportHeight, threshold)) {
                        LazyLoad.loadElement(el);
                    }
                });
            }
        },

        isInViewport: function(el, viewportHeight, threshold) {
            const rect = el.getBoundingClientRect();
            return (
                rect.top <= viewportHeight + threshold &&
                rect.bottom >= -threshold &&
                rect.left <= (window.innerWidth || document.documentElement.clientWidth) + threshold &&
                rect.right >= -threshold
            );
        },

        loadElement: function(el) {
            if (this.loaded.has(el)) {
                return;
            }

            this.loaded.add(el);

            const tagName = el.tagName.toLowerCase();

            if (tagName === 'img') {
                this.loadImage(el);
            } else if (tagName === 'iframe') {
                this.loadIframe(el);
            }
        },

        loadImage: function(img) {
            const src = img.dataset.src;
            const srcset = img.dataset.srcset;
            const sizes = img.dataset.sizes;

            if (!src) {
                return;
            }

            // Preload image
            const tempImg = new Image();
            
            tempImg.onload = function() {
                img.src = src;
                
                if (srcset) {
                    img.srcset = srcset;
                }
                
                if (sizes) {
                    img.sizes = sizes;
                }

                img.classList.add('enbombas-loaded');
                img.classList.remove('enbombas-lazy');

                // Remove data attributes
                delete img.dataset.src;
                delete img.dataset.srcset;
                delete img.dataset.sizes;

                // Trigger custom event
                img.dispatchEvent(new CustomEvent('enbombas:loaded'));
            };

            tempImg.onerror = function() {
                // Still try to load the image
                img.src = src;
                img.classList.add('enbombas-error');
            };

            tempImg.src = src;
        },

        loadIframe: function(iframe) {
            const src = iframe.dataset.src;

            if (!src) {
                return;
            }

            iframe.src = src;
            iframe.classList.add('enbombas-loaded');
            iframe.classList.remove('enbombas-lazy-iframe');

            delete iframe.dataset.src;

            // Trigger custom event
            iframe.dispatchEvent(new CustomEvent('enbombas:loaded'));
        },

        // Manual load function for external use
        load: function(element) {
            if (element) {
                this.loadElement(element);
            }
        },

        // Refresh observer for dynamically added content
        refresh: function() {
            if (this.observer) {
                this.observeElements();
            } else {
                this.checkVisibleElements();
            }
        }
    };

    /**
     * Delayed JS Module
     */
    const DelayedJS = {
        loaded: false,
        scripts: [],

        init: function() {
            if (!config.delayJS.enabled) {
                return;
            }

            // Collect delayed scripts
            this.collectScripts();

            // Bind trigger events
            this.bindTriggers();
        },

        collectScripts: function() {
            // Collect external scripts marked for delay
            document.querySelectorAll('script[type="text/enbombas-delay"][data-src]').forEach(function(el) {
                DelayedJS.scripts.push({
                    type: 'external',
                    src: el.dataset.src,
                    element: el
                });
            });

            // Collect inline scripts marked for delay
            document.querySelectorAll('script[type="text/enbombas-delay"]:not([data-src])').forEach(function(el) {
                DelayedJS.scripts.push({
                    type: 'inline',
                    content: el.textContent,
                    element: el
                });
            });
        },

        bindTriggers: function() {
            const self = this;
            const events = ['scroll', 'click', 'mousemove', 'keydown', 'touchstart', 'touchmove'];

            events.forEach(function(eventName) {
                window.addEventListener(eventName, function handler() {
                    window.removeEventListener(eventName, handler);
                    self.loadDelayed();
                }, { once: true, passive: true });
            });

            // Fallback timeout
            setTimeout(function() {
                self.loadDelayed();
            }, config.delayJS.timeout);
        },

        loadDelayed: function() {
            if (this.loaded) {
                return;
            }

            this.loaded = true;

            // Load scripts in order
            this.scripts.forEach(function(script, index) {
                setTimeout(function() {
                    DelayedJS.loadScript(script);
                }, index * 50); // Small delay between scripts
            });

            // Trigger custom event
            document.dispatchEvent(new CustomEvent('enbombas:delayed-loaded'));
        },

        loadScript: function(script) {
            if (script.type === 'external') {
                const newScript = document.createElement('script');
                newScript.src = script.src;
                newScript.async = true;
                document.body.appendChild(newScript);
            } else if (script.type === 'inline') {
                try {
                    const newScript = document.createElement('script');
                    newScript.textContent = script.content;
                    document.body.appendChild(newScript);
                } catch (e) {
                    console.error('EnBombas Nitro: Error executing delayed script', e);
                }
            }

            // Remove original element
            if (script.element && script.element.parentNode) {
                script.element.parentNode.removeChild(script.element);
            }
        }
    };

    /**
     * Initialize when DOM is ready
     */
    function init() {
        LazyLoad.init();
        DelayedJS.init();
    }

    // Initialize
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Expose to global scope for external use
    window.EnBombasNitro = {
        lazyload: LazyLoad,
        delayedJS: DelayedJS,
        refresh: function() {
            LazyLoad.refresh();
        },
        loadElement: function(el) {
            LazyLoad.load(el);
        }
    };

})();



