/**
 * En Bombas – Optimizador de Imágenes con IA
 * Admin JavaScript
 */

(function($) {
    'use strict';

    // Main EnBombas Admin Object
    const EnBombas = {
        // Bulk optimization state
        bulkState: {
            total: 0,
            processed: 0,
            success: 0,
            failed: 0,
            totalSaved: 0,
            running: false
        },

        /**
         * Initialize the admin functionality
         */
        init: function() {
            this.bindEvents();
            this.initAnimations();
            this.initProgressRings();
            this.initRadioCards();
            this.initCircularProgress();
        },

        /**
         * Bind all event handlers
         */
        bindEvents: function() {
            // Single image optimization
            $(document).on('click', '.enbombas-optimize, .enbombas-reoptimize', this.handleOptimize.bind(this));
            
            // Image restore
            $(document).on('click', '.enbombas-restore', this.handleRestore.bind(this));
            
            // Circular bulk button
            $(document).on('click', '#enb-bulk-button', this.handleBulkButtonClick.bind(this));
            
            // Stop button
            $(document).on('click', '#enbombas-stop-bulk', this.stopBulkOptimization.bind(this));
            
            // Settings form
            $(document).on('submit', '#enbombas-settings-form', this.handleSettingsSave.bind(this));
            
            // Toggle switches
            $(document).on('change', '.enbombas-toggle-input', this.handleToggle.bind(this));
            
            // Custom folders
            $(document).on('click', '.enbombas-folder-add-btn', this.addCustomFolder.bind(this));
            $(document).on('click', '.enbombas-folder-remove', this.removeCustomFolder.bind(this));
            
            // Radio cards
            $(document).on('click', '.enbombas-radio-card', this.selectRadioCard.bind(this));
            
            // Refresh stats
            $(document).on('click', '.enbombas-refresh-stats', this.refreshStats.bind(this));
            
            // Run monitoring scan
            $(document).on('click', '.enbombas-run-scan', this.runMonitoringScan.bind(this));
        },

        /**
         * Initialize animations
         */
        initAnimations: function() {
            // Animate stat numbers on page load
            $('.enbombas-stat-value, .enbombas-bulk-count').each(function() {
                const $el = $(this);
                const value = $el.text();
                
                // Only animate numeric values
                if (/^\d+/.test(value)) {
                    const numericValue = parseInt(value.replace(/[^0-9]/g, ''));
                    const suffix = value.replace(/[0-9]/g, '');
                    
                    $el.text('0' + suffix);
                    
                    $({ count: 0 }).animate({ count: numericValue }, {
                        duration: 1500,
                        easing: 'swing',
                        step: function() {
                            $el.text(Math.floor(this.count) + suffix);
                        },
                        complete: function() {
                            $el.text(numericValue + suffix);
                        }
                    });
                }
            });

            // Add energy pulse background if on dashboard
            if ($('.enbombas-admin-wrap').length && !$('.enbombas-energy-bg').length) {
                const energyBg = $('<div class="enbombas-energy-bg">' +
                    '<div class="enbombas-energy-pulse"></div>' +
                    '<div class="enbombas-energy-pulse"></div>' +
                    '<div class="enbombas-energy-pulse"></div>' +
                    '</div>');
                $('body').append(energyBg);
            }
        },

        /**
         * Initialize progress rings (for health score)
         */
        initProgressRings: function() {
            $('.enbombas-score-ring').each(function() {
                const $ring = $(this);
                const score = parseInt($ring.data('score')) || 0;
                const $progress = $ring.find('.progress');
                
                // Calculate circumference (r=42)
                const circumference = 2 * Math.PI * 42;
                const offset = circumference - (score / 100) * circumference;
                
                $progress.css({
                    'stroke-dasharray': circumference,
                    'stroke-dashoffset': circumference
                });
                
                // Animate the progress
                setTimeout(function() {
                    $progress.css('stroke-dashoffset', offset);
                }, 300);
            });
        },

        /**
         * Initialize radio cards
         */
        initRadioCards: function() {
            $('.enbombas-radio-card input:checked').each(function() {
                $(this).closest('.enbombas-radio-card').addClass('selected');
            });
        },

        /**
         * Initialize circular progress button
         */
        initCircularProgress: function() {
            const $button = $('#enb-bulk-button');
            if (!$button.length) return;

            // Get initial values from data attributes or elements
            const total = parseInt($button.data('total')) || parseInt($('.enbombas-bulk-count').first().text()) || 0;
            const processed = parseInt($('#enbombas-bulk-processed').text()) || 0;

            this.bulkState.total = total;
            this.bulkState.processed = processed;

            // Update UI
            this.updateCircularProgress(0);
        },

        /**
         * Update circular progress ring
         */
        updateCircularProgress: function(percent) {
            const $ring = $('.enb-ring-progress');
            const $percent = $('.enb-bulk-percent');
            
            if (!$ring.length) return;

            const circumference = 2 * Math.PI * 54; // r=54
            const offset = circumference - (percent / 100) * circumference;
            
            $ring.css('stroke-dashoffset', offset);
            $percent.text(percent + '%');
        },

        /**
         * Handle circular bulk button click
         */
        handleBulkButtonClick: function(e) {
            e.preventDefault();
            
            const $button = $('#enb-bulk-button');
            
            if ($button.hasClass('enb-bulk-running')) {
                // Already running, do nothing (use stop button)
                return;
            }

            if ($button.hasClass('enb-bulk-complete')) {
                // Reset and allow restart
                $button.removeClass('enb-bulk-complete');
                this.bulkState = {
                    total: 0,
                    processed: 0,
                    success: 0,
                    failed: 0,
                    totalSaved: 0,
                    running: false
                };
                location.reload();
                return;
            }

            // Start optimization
            this.startBulkOptimization();
        },

        /**
         * Handle single image optimization
         */
        handleOptimize: function(e) {
            e.preventDefault();
            
            const $btn = $(e.currentTarget);
            const attachmentId = $btn.data('id');
            const $container = $btn.closest('.enbombas-media-status, .enbombas-attachment-field');
            
            // Disable button and show loading
            $btn.prop('disabled', true);
            $btn.find('.enbombas-btn-text').hide();
            $btn.find('.enbombas-btn-loader').show();
            
            if (!$btn.find('.enbombas-btn-loader').length) {
                $btn.text(enbombasAdmin.strings.optimizing);
            }
            
            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'enbombas_optimize_single',
                    nonce: enbombasAdmin.nonce,
                    attachment_id: attachmentId
                },
                success: function(response) {
                    if (response.success) {
                        EnBombas.showToast('success', response.data.message);
                        
                        // Reload the column content
                        if ($container.hasClass('enbombas-media-status')) {
                            location.reload();
                        }
                    } else {
                        EnBombas.showToast('error', response.data.message || enbombasAdmin.strings.error);
                        $btn.prop('disabled', false);
                        $btn.find('.enbombas-btn-text').show();
                        $btn.find('.enbombas-btn-loader').hide();
                    }
                },
                error: function() {
                    EnBombas.showToast('error', enbombasAdmin.strings.error);
                    $btn.prop('disabled', false);
                    $btn.find('.enbombas-btn-text').show();
                    $btn.find('.enbombas-btn-loader').hide();
                }
            });
        },

        /**
         * Handle image restore
         */
        handleRestore: function(e) {
            e.preventDefault();
            
            if (!confirm(enbombasAdmin.strings.confirm_restore)) {
                return;
            }
            
            const $btn = $(e.currentTarget);
            const attachmentId = $btn.data('id');
            
            $btn.prop('disabled', true).text(enbombasAdmin.strings.restoring);
            
            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'enbombas_restore_single',
                    nonce: enbombasAdmin.nonce,
                    attachment_id: attachmentId
                },
                success: function(response) {
                    if (response.success) {
                        EnBombas.showToast('success', response.message || enbombasAdmin.strings.restored);
                        location.reload();
                    } else {
                        EnBombas.showToast('error', response.message || enbombasAdmin.strings.error);
                        $btn.prop('disabled', false).text('Restaurar');
                    }
                },
                error: function() {
                    EnBombas.showToast('error', enbombasAdmin.strings.error);
                    $btn.prop('disabled', false).text('Restaurar');
                }
            });
        },

        /**
         * Start bulk optimization
         */
        startBulkOptimization: function() {
            const $button = $('#enb-bulk-button');
            const $stopBtn = $('#enbombas-stop-bulk');
            const $label = $button.find('.enb-bulk-label');
            
            // Update UI
            $button.addClass('enb-bulk-running').prop('disabled', false);
            $stopBtn.show();
            $label.text(enbombasAdmin.strings.bulk_processing || 'Optimizando imágenes...');
            
            this.bulkState.running = true;
            
            // Initialize bulk optimization
            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'enbombas_bulk_optimize',
                    nonce: enbombasAdmin.nonce,
                    bulk_action: 'start'
                },
                success: function(response) {
                    if (response.success) {
                        // Get total from response
                        if (response.total) {
                            EnBombas.bulkState.total = response.total;
                        }
                        EnBombas.showToast('success', enbombasAdmin.strings.bulk_starting);
                        EnBombas.processBulkBatch();
                    } else {
                        EnBombas.showToast('error', response.message || enbombasAdmin.strings.error);
                        EnBombas.resetBulkUI();
                    }
                },
                error: function() {
                    EnBombas.showToast('error', enbombasAdmin.strings.error);
                    EnBombas.resetBulkUI();
                }
            });
        },

        /**
         * Process a batch of images
         */
        processBulkBatch: function() {
            if (!this.bulkState.running) return;

            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'enbombas_bulk_optimize',
                    nonce: enbombasAdmin.nonce,
                    bulk_action: 'process'
                },
                success: function(response) {
                    if (response.success) {
                        const status = response.status || response.data?.status || {};
                        
                        // Update state from response
                        EnBombas.bulkState.processed = status.processed || 0;
                        EnBombas.bulkState.success = status.success || 0;
                        EnBombas.bulkState.failed = status.failed || 0;
                        EnBombas.bulkState.totalSaved = status.total_saved || 0;
                        
                        if (status.total) {
                            EnBombas.bulkState.total = status.total;
                        }
                        
                        // Calculate and update progress
                        EnBombas.updateBulkProgress(status);
                        
                        if (response.complete || response.data?.complete) {
                            EnBombas.completeBulkOptimization(status);
                        } else if (EnBombas.bulkState.running) {
                            // Continue processing
                            setTimeout(function() {
                                EnBombas.processBulkBatch();
                            }, 500);
                        }
                    } else {
                        EnBombas.showToast('error', response.message || enbombasAdmin.strings.error);
                        EnBombas.resetBulkUI();
                    }
                },
                error: function() {
                    EnBombas.showToast('error', enbombasAdmin.strings.error);
                    EnBombas.resetBulkUI();
                }
            });
        },

        /**
         * Update bulk progress UI
         */
        updateBulkProgress: function(status) {
            const total = status.total || this.bulkState.total || 1;
            const processed = status.processed || this.bulkState.processed || 0;
            const success = status.success || this.bulkState.success || 0;
            const failed = status.failed || this.bulkState.failed || 0;
            const saved = status.total_saved_formatted || size_format(status.total_saved || 0);
            
            // Calculate percentage
            const percent = total > 0 ? Math.round((processed / total) * 100) : 0;
            
            // Update circular progress
            this.updateCircularProgress(percent);
            
            // Update stats
            $('#enbombas-bulk-processed').text(processed);
            $('#enbombas-bulk-success').text(success);
            $('#enbombas-bulk-failed').text(failed);
            $('#enbombas-bulk-saved').text(saved);
        },

        /**
         * Complete bulk optimization
         */
        completeBulkOptimization: function(status) {
            const $button = $('#enb-bulk-button');
            const $label = $button.find('.enb-bulk-label');
            
            this.bulkState.running = false;
            
            // Update UI
            $button.removeClass('enb-bulk-running').addClass('enb-bulk-complete');
            $label.text(enbombasAdmin.strings.bulk_complete || 'Optimización completada');
            this.updateCircularProgress(100);
            
            $('#enbombas-stop-bulk').hide();
            
            EnBombas.showToast('success', enbombasAdmin.strings.bulk_complete);
            
            // Update subtitle
            const successCount = status?.success || this.bulkState.success || 0;
            const savedFormatted = status?.total_saved_formatted || '0 B';
            $('.enbombas-bulk-subtitle').text(
                'Optimización completada. ' + successCount + ' imágenes optimizadas, ' + savedFormatted + ' ahorrados.'
            );
        },

        /**
         * Stop bulk optimization
         */
        stopBulkOptimization: function(e) {
            if (e) e.preventDefault();
            
            this.bulkState.running = false;
            
            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'enbombas_bulk_optimize',
                    nonce: enbombasAdmin.nonce,
                    bulk_action: 'stop'
                },
                success: function(response) {
                    EnBombas.showToast('success', enbombasAdmin.strings.bulk_stopped);
                    EnBombas.resetBulkUI();
                },
                error: function() {
                    EnBombas.resetBulkUI();
                }
            });
        },

        /**
         * Reset bulk UI
         */
        resetBulkUI: function() {
            const $button = $('#enb-bulk-button');
            const $label = $button.find('.enb-bulk-label');
            
            this.bulkState.running = false;
            
            $button.removeClass('enb-bulk-running enb-bulk-complete').prop('disabled', false);
            $label.text('Empezar optimización');
            $('#enbombas-stop-bulk').hide();
        },

        /**
         * Handle settings save
         */
        handleSettingsSave: function(e) {
            e.preventDefault();
            
            const $form = $(e.currentTarget);
            const $btn = $form.find('button[type="submit"]');
            
            $btn.prop('disabled', true);
            
            const formData = new FormData($form[0]);
            formData.append('action', 'enbombas_save_settings');
            formData.append('nonce', enbombasAdmin.nonce);
            
            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        EnBombas.showToast('success', response.data.message);
                    } else {
                        EnBombas.showToast('error', response.data.message || enbombasAdmin.strings.error);
                    }
                    $btn.prop('disabled', false);
                },
                error: function() {
                    EnBombas.showToast('error', enbombasAdmin.strings.error);
                    $btn.prop('disabled', false);
                }
            });
        },

        /**
         * Handle toggle change
         */
        handleToggle: function(e) {
            const $toggle = $(e.currentTarget);
            const $dependent = $('[data-depends-on="' + $toggle.attr('name') + '"]');
            
            if ($toggle.is(':checked')) {
                $dependent.slideDown(200);
            } else {
                $dependent.slideUp(200);
            }
        },

        /**
         * Select radio card
         */
        selectRadioCard: function(e) {
            const $card = $(e.currentTarget);
            const $container = $card.closest('.enbombas-radio-cards');
            
            $container.find('.enbombas-radio-card').removeClass('selected');
            $card.addClass('selected');
            $card.find('input[type="radio"]').prop('checked', true);
        },

        /**
         * Add custom folder
         */
        addCustomFolder: function(e) {
            e.preventDefault();
            
            const $input = $('#enbombas-new-folder');
            const path = $input.val().trim();
            
            if (!path) {
                return;
            }
            
            const $list = $('.enbombas-folders-list');
            const $item = $('<div class="enbombas-folder-item">' +
                '<span class="enb-icon"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"/></svg></span>' +
                '<span class="enbombas-folder-path">' + EnBombas.escapeHtml(path) + '</span>' +
                '<input type="hidden" name="custom_folders[]" value="' + EnBombas.escapeHtml(path) + '">' +
                '<button type="button" class="enbombas-folder-remove">×</button>' +
                '</div>');
            
            // Remove "no folders" message if present
            $list.find('p').remove();
            
            $list.append($item);
            $input.val('');
        },

        /**
         * Remove custom folder
         */
        removeCustomFolder: function(e) {
            e.preventDefault();
            $(e.currentTarget).closest('.enbombas-folder-item').remove();
        },

        /**
         * Refresh stats
         */
        refreshStats: function(e) {
            e.preventDefault();
            
            const $btn = $(e.currentTarget);
            $btn.addClass('spinning');
            
            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'enbombas_get_stats',
                    nonce: enbombasAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    }
                    $btn.removeClass('spinning');
                },
                error: function() {
                    $btn.removeClass('spinning');
                }
            });
        },

        /**
         * Run monitoring scan
         */
        runMonitoringScan: function(e) {
            e.preventDefault();
            
            const $btn = $(e.currentTarget);
            const originalHtml = $btn.html();
            $btn.prop('disabled', true).html('<span class="enb-icon"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" class="spinning"><circle cx="12" cy="12" r="10"/></svg></span> Escaneando...');
            
            $.ajax({
                url: enbombasAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'enbombas_get_monitoring_data',
                    nonce: enbombasAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        EnBombas.showToast('success', 'Escaneo completado');
                        location.reload();
                    } else {
                        EnBombas.showToast('error', response.data?.message || 'Error en el escaneo');
                    }
                    $btn.prop('disabled', false).html(originalHtml);
                },
                error: function() {
                    EnBombas.showToast('error', 'Error en el escaneo');
                    $btn.prop('disabled', false).html(originalHtml);
                }
            });
        },

        /**
         * Show toast notification
         */
        showToast: function(type, message) {
            // Remove existing toasts
            $('.enbombas-toast').remove();
            
            const iconSvg = type === 'success' 
                ? '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>'
                : '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>';
            
            const $toast = $('<div class="enbombas-toast enbombas-toast-' + type + '">' +
                '<span class="enbombas-toast-icon"><span class="enb-icon">' + iconSvg + '</span></span>' +
                '<span class="enbombas-toast-message">' + message + '</span>' +
                '<button class="enbombas-toast-close">×</button>' +
                '</div>');
            
            $('body').append($toast);
            
            // Auto-remove after 5 seconds
            setTimeout(function() {
                $toast.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
            
            // Close button
            $toast.find('.enbombas-toast-close').on('click', function() {
                $toast.fadeOut(300, function() {
                    $(this).remove();
                });
            });
        },

        /**
         * Escape HTML entities
         */
        escapeHtml: function(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Simple size format helper
    function size_format(bytes) {
        if (bytes === 0) return '0 B';
        const k = 1024;
        const sizes = ['B', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
    }

    // Initialize on document ready
    $(document).ready(function() {
        EnBombas.init();
    });

    // Expose globally for external use
    window.EnBombas = EnBombas;

})(jQuery);
